use crate::{
    cargo::CargoRanges,
    sonar::{Issue, IssueType, Issues, Location, Severity, TextRange},
};
use eyre::{Context, Result};
use rustsec::{Report, Vulnerability};
use std::process::{Command, Stdio};
use tracing::{debug, error, info, trace};

const AUDIT_ENGINE: &str = "audit";

pub struct Audit<'ranges> {
    cargo_ranges: &'ranges CargoRanges,
}

impl<'ranges> Audit<'ranges> {
    pub fn new(cargo_ranges: &'ranges CargoRanges) -> Self {
        Self { cargo_ranges }
    }

    // FIXME: reporting on 'Cargo.toml' is a bad idea because the vulnerability might exist on a nested dependency
    // it might be possible to use 'Cargo.lock' instead (or use 'cargo-deny' to obtain a graph of dependency and find the root dependency)
    fn to_issue(&self, vulnerability: &Vulnerability) -> Issue {
        let advisory = &vulnerability.advisory;
        let rule_id = Some(advisory.id.to_string());
        let message = format!(
            "{} (see https://github.com/rustsec/advisory-db/blob/main/crates/{}/{}.md)",
            advisory.title, advisory.package, advisory.id
        );
        let file_path = self.cargo_ranges.cargo_path.to_string_lossy().to_string();
        let text_range = self
            .cargo_ranges
            .dependencies
            .get(advisory.package.as_str())
            .and_then(|ranges| ranges.1.as_ref())
            .or_else(|| {
                self.cargo_ranges
                    .dependencies
                    .get(advisory.package.as_str())
                    .map(|ranges| &ranges.0)
            })
            .map(|range| TextRange::clone(range))
            .unwrap_or_else(|| {
                error!("failed to find a corresponding text range for this vulnerability. This is probably a bug in `cargo-sonar`, if you can please report an issue, we'd be happy to try and fix it.");
                TextRange {
                    start_line: 1,
                    end_line: 1,
                    start_column: 0,
                    end_column: 1,
                }
            });
        let primary_location = Location {
            message,
            file_path,
            text_range,
        };
        let secondary_locations = Vec::new();
        Issue {
            engine_id: AUDIT_ENGINE.to_string(),
            rule_id,
            severity: Severity::Major,
            r#type: IssueType::Vulnerability,
            primary_location,
            secondary_locations,
        }
    }
    pub fn to_issues(&self, report: Report) -> Issues {
        report
            .vulnerabilities
            .list
            .iter()
            .map(|vulnerability| self.to_issue(vulnerability))
            .collect()
    }
}

impl std::convert::TryInto<Issues> for Audit<'_> {
    type Error = eyre::Error;

    fn try_into(self) -> Result<Issues> {
        trace!("prepare `cargo clippy` command");
        // FIXME: replace with 'cargo-deny' which provides more functionalities and a graph of dependencies
        let mut cmd = Command::new("cargo");
        cmd.arg("audit")
            .arg("--json")
            .stdout(Stdio::piped())
            .stderr(Stdio::null());
        debug!("execute `{:?}`", cmd);
        let child_stdout = cmd.spawn()?.stdout;
        let issues = if let Some(mut stdout) = child_stdout {
            let mut output = String::new();
            std::io::Read::read_to_string(&mut stdout, &mut output)?;
            let report = serde_json::from_str::<Report>(&output)
                .context("failed to be parsed as a 'rustsec::report::Report'")?;
            self.to_issues(report)
        } else {
            Issues::default()
        };
        info!("{} sonar issues created", issues.len());
        Ok(issues)
    }
}

impl From<Report> for Issues {
    fn from(_: Report) -> Self {
        Self::default()
    }
}
