use crate::sonar::{Issue, IssueType, Issues, Location, Severity, TextRange};
use cargo_metadata::{
    diagnostic::{DiagnosticLevel, DiagnosticSpan},
    Message,
};
use skip_error::SkipError as _;
use std::{
    io::{BufRead as _, BufReader},
    process::{Command, Stdio},
};
use tracing::{debug, info, trace};

const CLIPPY_ENGINE: &str = "clippy";

#[derive(Debug)]
pub struct Clippy;

#[derive(Debug, thiserror::Error)]
pub enum Error {
    #[error("no span defined to report an error")]
    NoSpan,
    #[error("the message of type '{0}' are not handled")]
    InvalidMessage(&'static str),
}

impl From<&DiagnosticLevel> for Severity {
    fn from(level: &DiagnosticLevel) -> Self {
        use DiagnosticLevel::*;
        match level {
            Ice => Severity::Blocker,
            Error => Severity::Critical,
            Warning => Severity::Major,
            FailureNote => Severity::Minor,
            Note | Help => Severity::Info,
            _ => Severity::Info,
        }
    }
}

impl From<&DiagnosticSpan> for TextRange {
    fn from(span: &DiagnosticSpan) -> Self {
        Self {
            start_line: span.line_start,
            end_line: span.line_end,
            start_column: span.column_start,
            // clippy does consider the end-of-line character
            // but sonar-scanner does not and check for validity
            end_column: span.column_end - 1,
        }
    }
}

impl TryFrom<Message> for Issue {
    type Error = Error;
    fn try_from(message: Message) -> Result<Self, Self::Error> {
        use Message::*;
        if let CompilerMessage(message) = message {
            let rule_id = message
                .message
                .code
                .as_ref()
                .map(|diagnostic_code| diagnostic_code.code.clone());
            let severity = Severity::from(&message.message.level);
            let r#type = IssueType::CodeSmell;
            let (primary_location, secondary_locations) = match message.message.spans.len() {
                0 => return Err(Error::NoSpan),
                n => {
                    let span = &message.message.spans[0];
                    let primary_location = Location {
                        message: message.message.message.clone(),
                        file_path: span.file_name.clone(),
                        text_range: TextRange::from(span),
                    };
                    let secondary_locations = (1..n)
                        .flat_map(|idx| message.message.spans.get(idx))
                        .map(|span| Location {
                            message: message.message.message.clone(),
                            file_path: span.file_name.clone(),
                            text_range: TextRange::from(span),
                        })
                        .collect();
                    (primary_location, secondary_locations)
                }
            };
            let issue = Self {
                engine_id: CLIPPY_ENGINE.to_string(),
                rule_id,
                severity,
                r#type,
                primary_location,
                secondary_locations,
            };
            Ok(issue)
        } else {
            let kind = match message {
                CompilerArtifact(_) => "compiler-artifact",
                CompilerMessage(_) => {
                    unreachable!("'CompilerMessage' has already been parsed above")
                }
                BuildScriptExecuted(_) => "build-script-executed",
                BuildFinished(_) => "build-finished",
                TextLine(_) => "text-line",
                _ => "unknown",
            };
            Err(Error::InvalidMessage(kind))
        }
    }
}

impl Clippy {
    pub fn new() -> Self {
        Self
    }
}

impl std::convert::TryInto<Issues> for Clippy {
    type Error = eyre::Error;

    fn try_into(self) -> Result<Issues, Self::Error> {
        trace!("prepare `cargo clippy` command");
        let mut cmd = Command::new("cargo");
        cmd.arg("clippy")
            .arg("--message-format=json")
            .stdout(Stdio::piped())
            .stderr(Stdio::null());
        debug!("execute `{:?}`", cmd);
        let issues: Issues = cmd
            .spawn()?
            .stdout
            .into_iter()
            .flat_map(|stdout| BufReader::new(stdout).lines())
            .flatten()
            .flat_map(|line| serde_json::from_str::<cargo_metadata::Message>(&line))
            .map(Issue::try_from)
            .skip_error_and_debug()
            .collect();
        info!("{} sonar issues created", issues.len());
        Ok(issues)
    }
}
