use crate::sonar::TextRange;
use cargo::{
    core::{source::SourceId, EitherManifest, Manifest},
    util::config::Config,
};
use eyre::{bail, eyre};
use regex::Regex;
use std::{
    collections::BTreeMap,
    fs::File,
    io::{BufRead as _, BufReader},
    path::{Path, PathBuf},
};
use tracing::{debug, instrument, trace};

#[derive(Debug)]
pub struct CargoRanges {
    pub cargo_path: PathBuf,
    pub manifest: Manifest,
    // key is the name of the dependency
    // tuple are text ranges for (dependency-name, dependency-version)
    pub dependencies: BTreeMap<String, (TextRange, Option<TextRange>)>,
}

impl std::convert::TryFrom<&Path> for CargoRanges {
    type Error = eyre::Error;

    #[instrument(level = "debug")]
    fn try_from(cargo_path: &Path) -> Result<Self, Self::Error> {
        let cargo_path = cargo_path.canonicalize()?;
        let config = Config::default().map_err(|e| eyre!(e))?;
        let source_path = cargo_path
            .parent()
            .ok_or_else(|| eyre!("failed to find source's folder of the manifest file"))?;
        let source_id = SourceId::for_path(source_path).map_err(|e| eyre!(e))?;
        let (manifest, _) = cargo::util::toml::read_manifest(&cargo_path, source_id, &config)?;
        debug!("manifest file parsed");
        if let EitherManifest::Real(manifest) = manifest {
            let mut dependencies = BTreeMap::new();
            let file = File::open(&cargo_path)?;
            let reader = BufReader::new(file);
            // FIXME: really dirty algorithm's complexity: for each line of the manifest,
            // look in all dependencies to see if one matches
            for (line_num, line) in reader.lines().enumerate() {
                let line_num = line_num + 1;
                let line = line?;
                let span = tracing::trace_span!("manifest", line_num = line_num, line = %line);
                let _enter = span.enter();
                for dependency in manifest.dependencies() {
                    let name = dependency.name_in_toml();
                    let version = dependency.version_req();
                    let name_pattern = format!("\\b{}\\b", name);
                    let version_pattern = version
                        .to_string()
                        // "0.56" is understood as "^0.56" by cargo's parser
                        .replace('^', "\\^?")
                        .replace('.', "\\.");
                    // FIXME: this works only for dependencies declared on one line
                    // This won't work for dependencies declared on multiple lines
                    // ```
                    // [dependencies.serde]
                    // version = "1.0"
                    // ```
                    let pattern = format!(
                        "^(?P<name>{})\\s+=.*\"(?P<version>{})\"",
                        name_pattern, version_pattern,
                    );
                    let pattern = Regex::new(&pattern)?;
                    let span = tracing::trace_span!("match", dependency = %name, version = %version, regex = %pattern);
                    let _enter = span.enter();
                    // FIXME: This will match only if both `name` and `version` are found
                    // however, we might still want to have an entry for `name`
                    // even if `version` can't be found
                    // (half of something is better than nothing at all)
                    if let Some(capture) = pattern.captures(&line) {
                        if let Some(name_match) = capture.name("name") {
                            let name = name_match.as_str().to_string();
                            trace!("found text ranges dependency's name");
                            let name_range = TextRange {
                                start_line: line_num,
                                end_line: line_num,
                                start_column: name_match.start(),
                                end_column: name_match.end(),
                            };
                            if let Some(version_match) = capture.name("version") {
                                trace!("found text ranges dependency's version");
                                let version_range = TextRange {
                                    start_line: line_num,
                                    end_line: line_num,
                                    start_column: version_match.start(),
                                    end_column: version_match.end(),
                                };
                                dependencies.insert(name, (name_range, Some(version_range)));
                            } else {
                                dependencies.insert(name, (name_range, None));
                            }
                        }
                    }
                }
            }
            Ok(Self {
                cargo_path,
                manifest,
                dependencies,
            })
        } else {
            trace!("manifest is not a real manifest, cannot do anything with it");
            bail!("failed to parse the cargo configuration");
        }
    }
}
