use crate::cargo::BinFile;
use crate::util::command;
use anyhow::Context;
use anyhow::Error;
use inferno::collapse::dtrace::Folder;
use inferno::collapse::dtrace::Options as CollapseOptions;
use inferno::collapse::Collapse;
use std::fs::OpenOptions;
use std::io::BufReader;
use std::path::Path;
use std::process::Command;

pub(crate) fn make_dtrace_command(
    root: bool,
    file: &BinFile,
    output: &Path,
    freq: Option<u32>,
    custom_cmd: Option<String>,
    args: &[String],
) -> Result<Command, Error> {
    let mut c = command(root, "dtrace");

    let dtrace_script = custom_cmd.unwrap_or(format!(
        "profile-{} /pid == $target/ {{ @[ustack(100)] = count(); }}",
        freq.unwrap_or(997)
    ));

    c.arg("-x");
    c.arg("ustackframes=100");

    c.arg("-n");
    c.arg(&dtrace_script);

    c.arg("-o");
    c.arg(output);

    let mut escaped = String::new();
    escaped.push_str(&file.path.to_string_lossy());
    for arg in args {
        escaped.push(' ');
        escaped.push_str(&arg.replace(" ", "\\ "));
    }
    if file.is_bench {
        escaped.push_str(" --bench");
    }

    c.arg("-c");
    c.arg(&escaped);

    Ok(c)
}

pub(crate) fn to_collapsed(stacks_file: &Path) -> Result<Vec<u8>, Error> {
    let output = OpenOptions::new()
        .read(true)
        .write(false)
        .open(stacks_file)
        .with_context(|| {
            format!(
                "failed to open stacks file ({}) generated by dtrace",
                stacks_file.display()
            )
        })?;
    let perf_reader = BufReader::new(&output);

    let mut collapsed = vec![];

    let collapse_options = CollapseOptions::default();

    Folder::from(collapse_options)
        .collapse(perf_reader, &mut collapsed)
        .with_context(|| {
            format!(
                "unable to collapse generated profile data from {}",
                stacks_file.display()
            )
        })?;

    Ok(collapsed)
}
