/// A quick and dirty parser for the .config files generated by kconfig systems like
/// the ESP-IDF one
use std::{
    convert::TryFrom,
    env, fs,
    io::{self, BufRead},
    path::Path,
    vec,
};

use anyhow::*;

const VAR_CFG_ARGS_KEY: &'static str = "CARGO_PIO_CFG_ARGS";

#[derive(Copy, Clone, PartialEq, Eq, Hash, Debug)]
pub enum Tristate {
    True,
    False,
    Module,
    NotSet,
}

#[derive(Clone, Debug)]
pub enum Value {
    Tristate(Tristate),
    String(String),
}

impl Value {
    fn parse(str: impl AsRef<str>) -> Option<Self> {
        let str = str.as_ref();

        Some(if str.starts_with('\"') {
            Self::String(str.to_owned()) // TODO: Properly parse and escape
        } else if str == "y" {
            Self::Tristate(Tristate::True)
        } else if str == "n" {
            Self::Tristate(Tristate::False)
        } else if str == "m" {
            Self::Tristate(Tristate::Module)
        } else {
            return None;
        })
    }
}

pub fn load(path: impl AsRef<Path>) -> Result<impl Iterator<Item = (String, Value)>> {
    Ok(io::BufReader::new(fs::File::open(path.as_ref())?)
        .lines()
        .filter_map(|line| line.ok().map(|l| l.trim().to_owned()))
        .filter(|line| !line.starts_with("#"))
        .filter_map(|line| {
            let mut split = line.split("=");

            if let Some(key) = split.next() {
                split
                    .next()
                    .map(|v| v.trim())
                    .map(Value::parse)
                    .flatten()
                    .map(|value| (key.to_owned(), value))
            } else {
                None
            }
        }))
}

#[derive(Clone, Debug)]
pub struct CfgArgs(vec::Vec<(String, Value)>);

impl TryFrom<&Path> for CfgArgs {
    type Error = anyhow::Error;

    fn try_from(path: &Path) -> Result<Self> {
        Ok(Self(load(path)?.collect()))
    }
}

impl CfgArgs {
    pub fn output(&self, prefix: impl AsRef<str>) {
        let prefix = prefix.as_ref();

        for arg in self.gather() {
            Self::output_cfg_arg(prefix, arg);
        }
    }

    pub fn propagate(&self) {
        let args = self.gather();

        output(VAR_CFG_ARGS_KEY, args.join(":"));
    }

    pub fn output_propagated(from_crate: impl AsRef<str>) -> Result<()> {
        let from_crate = from_crate.as_ref();

        for arg in split(env::var(format!(
            "DEP_{}_{}",
            from_crate, VAR_CFG_ARGS_KEY
        ))?) {
            Self::output_cfg_arg(from_crate, arg);
        }

        Ok(())
    }

    pub fn output_cfg_arg(prefix: impl AsRef<str>, arg: impl AsRef<str>) {
        println!(
            "cargo:rustc-cfg={}_{}",
            prefix.as_ref().to_lowercase(),
            arg.as_ref().to_lowercase()
        );
    }

    pub fn gather(&self) -> vec::Vec<String> {
        let mut result = Vec::new();

        for (key, value) in &self.0 {
            match value {
                Value::Tristate(Tristate::True) => result.push(key.clone()),
                _ => (),
            }
        }

        result
    }
}

fn output(key: impl AsRef<str>, value: impl AsRef<str>) {
    println!("cargo:{}={}", key.as_ref(), value.as_ref());
}

fn split(arg: impl AsRef<str>) -> Vec<String> {
    arg.as_ref()
        .split(":")
        .map(str::to_owned)
        .collect::<Vec<String>>()
}
