// SPDX-License-Identifier: GPL-3.0-or-later

use clap::{Parser, Subcommand};
use nix::unistd::Uid;
use std::{env::current_dir, path::Path, process::Command};

pub mod descriptor;
pub use descriptor::*;
pub mod build;
pub use build::*;

#[derive(Parser)]
#[clap(name = "cargo")]
#[clap(bin_name = "cargo")]
#[clap(override_usage = "cargo gra <SUBCOMMAND>")]
struct Cargo {
    // take the self name (gra) argument provided by cargo when calling via cargo gra ...
    #[clap(help = "Own subcommand name (gra)")]
    _self: Option<String>,
    #[clap(subcommand)]
    gra: Gra,
}

#[derive(Subcommand)]
#[clap(author, version, about)]
enum Gra {
    #[clap(about = "Cleanup generated files")]
    Clean,
    #[clap(
        about = "Generate manifests and source files from your Cargo.toml",
        override_usage = "cargo gra generate"
    )]
    Generate,
    #[clap(
        about = "Alias for cargo gra generate",
        override_usage = "cargo gra gen"
    )]
    Gen,
    #[clap(
        name = "install-gsettings",
        alias = "igs",
        about = "Install your apps gsettings schema (Requires sudo).",
        long_about = "Does the same as `install -D target/gra-gen/<app-id>.gschema.xml /usr/share/glib-2.0/schemas && glib-compile-schemas /usr/share/glib-2.0/schemas`",
        override_usage = "cargo gra install-gsettings"
    )]
    InstallGSettings,
    #[clap(
        name = "uninstall-gsettings",
        alias = "ugs",
        about = "Uninstall your apps gsettings schema",
        long_about = "Does the same as `rm /usr/share/glib-2.0/schemas/<app-id>.gschema.xml && glib-compile-schemas /usr/share/glib-2.0/schemas`",
        override_usage = "cargo gra uninstall-gsettings"
    )]
    UninstallGSettings,
    #[clap(
        name = "flatpak",
        about = "Build a flatpak app locally",
        override_usage = "cargo gra flatpak"
    )]
    Flatpak(Flatpak),
}

#[derive(Parser)]
struct Flatpak {
    #[clap(
        short,
        long,
        help = "Create a release tar.xz of the sources and manifest.yml with according hash and the given url. Provide 'local' to generate files for a local release."
    )]
    release: Option<String>,
    #[clap(
        short,
        long,
        help = "Only prepare the flatpak-temp. May be used to get everything set up and use flatpak-builder on your own."
    )]
    prepare: bool,
    #[clap(
        short,
        long,
        help = "The architecture to build for. The systems architecture will be used if unset. See https://docs.flatpak.org/en/latest/flatpak-builder-command-reference.html."
    )]
    arch: Option<String>,
}

impl From<Flatpak> for FlatpakArgs {
    fn from(f: Flatpak) -> Self {
        Self {
            arch: f.arch,
            prepare_only: f.prepare,
            release: f.release,
        }
    }
}

const GEN_DIR: &str = "gra-gen";
const GSETTINGS_SCHEMAS_DIR: &str = "/usr/share/glib-2.0/schemas";

fn main() {
    let c: Cargo = Cargo::parse();

    let project_dir = current_dir().unwrap();
    let target_dir = target_dir();

    match c.gra {
        Gra::Clean => clean(&project_dir),
        Gra::Generate => generate(&project_dir),
        Gra::Gen => generate(&project_dir),
        Gra::InstallGSettings => {
            if !Uid::effective().is_root() {
                println!("You must run install-gsettings with root permissions.");
                return;
            }

            let gra_gen = target_dir.join(GEN_DIR);
            if let Ok(descriptor) = parse_project_descriptor(Path::new("Cargo.toml")) {
                let o = Command::new("install")
                    .arg("-D")
                    .arg(format!(
                        "{}/{}.gschema.xml",
                        gra_gen.to_str().unwrap(),
                        &descriptor.app.id
                    ))
                    .arg(GSETTINGS_SCHEMAS_DIR)
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    if !a.trim().is_empty() {
                        eprintln!("[gra] {}", a);
                        return;
                    }
                }
                let o = Command::new("glib-compile-schemas")
                    .arg(GSETTINGS_SCHEMAS_DIR)
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    if !a.trim().is_empty() {
                        eprintln!("[gra] {}", a);
                        return;
                    }
                }
                println!("[gra] Installed gsettings in {}.", GSETTINGS_SCHEMAS_DIR);
            } else {
                println!("[gra] Could not parse project descriptor.");
            }
        }
        Gra::UninstallGSettings => {
            if let Ok(descriptor) = parse_project_descriptor(Path::new("Cargo.toml")) {
                let o = Command::new("rm")
                    .arg(format!(
                        "{}/{}.gschema.xml",
                        GSETTINGS_SCHEMAS_DIR, &descriptor.app.id
                    ))
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    eprintln!("[gra] {}", a);
                }
                let o = Command::new("glib-compile-schemas")
                    .arg(GSETTINGS_SCHEMAS_DIR)
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    eprintln!("[gra] {}", a);
                }
            } else {
                println!("[gra] Could not parse project descriptor.");
            }
        }
        Gra::Flatpak(command) => {
            if let Err(e) = flatpak(&project_dir, command.into()) {
                eprintln!("[gra] Failed to build flatpak: {}.", e);
            }
        }
    }
}
