// SPDX-License-Identifier: GPL-3.0-or-later

use clap::{Parser, Subcommand};
use fs_extra::file::read_to_string;
use gtk_rust_app::parse_project_descriptor;
use nix::unistd::Uid;
use std::{
    fs::{create_dir_all, remove_dir_all, remove_file, File},
    io::Write,
    path::{Path, PathBuf},
    process::Command,
};

#[derive(Parser)]
#[clap(name = "cargo")]
#[clap(bin_name = "cargo")]
#[clap(override_usage = "cargo gra <SUBCOMMAND>")]
struct Cargo {
    // take the self name (gra) argument provided by cargo when calling via cargo gra ...
    #[clap(help = "Own subcommand name (gra)")]
    _self: Option<String>,
    #[clap(subcommand)]
    gra: Gra,
}

#[derive(Subcommand)]
#[clap(author, version, about)]
enum Gra {
    #[clap(about = "Cleanup generated files")]
    Clean,
    #[clap(
        about = "Generate manifests and source files from your Cargo.toml",
        override_usage = "cargo gra generate"
    )]
    Generate,
    #[clap(
        about = "Alias for cargo gra generate",
        override_usage = "cargo gra gen"
    )]
    Gen,
    #[clap(
        name = "install-gsettings",
        alias = "igs",
        about = "Install your apps gsettings schema (Requires sudo).",
        long_about = "Does the same as `install -D target/gra-gen/<app-id>.gschema.xml /usr/share/glib-2.0/schemas && glib-compile-schemas /usr/share/glib-2.0/schemas`",
        override_usage = "cargo gra install-gsettings"
    )]
    InstallGSettings,
    #[clap(
        name = "uninstall-gsettings",
        alias = "ugs",
        about = "Uninstall your apps gsettings schema",
        long_about = "Does the same as `rm /usr/share/glib-2.0/schemas/<app-id>.gschema.xml && glib-compile-schemas /usr/share/glib-2.0/schemas`",
        override_usage = "cargo gra uninstall-gsettings"
    )]
    UninstallGSettings,
    #[clap(
        name = "flatpak",
        about = "Build a flatpak app locally",
        override_usage = "cargo gra flatpak"
    )]
    Flatpak(Flatpak),
}

#[derive(Parser)]
struct Flatpak {
    #[clap(
        short,
        long,
        help = "Create a release tar.xz of the sources and manifest.yml with according hash and the given url"
    )]
    release: Option<String>,
    #[clap(
        short,
        long,
        help = "Only prepare the flatpak-temp. May be used to get everything set up and use flatpak-builder on your own."
    )]
    prepare: bool,
    #[clap(
        short,
        long,
        help = "The architecture to build for. The systems architecture will be used if unset. See https://docs.flatpak.org/en/latest/flatpak-builder-command-reference.html."
    )]
    arch: Option<String>,
}

const GEN_DIR: &str = "gra-gen";
const GSETTINGS_SCHEMAS_DIR: &str = "/usr/share/glib-2.0/schemas";

fn main() {
    let c: Cargo = Cargo::parse();

    let target_env = std::env::var("CARGO_TARGET_DIR").unwrap_or("target".into());
    let target_dir = PathBuf::new().join(target_env);

    match c.gra {
        Gra::Clean => clean(target_dir),
        Gra::Generate => setup(target_dir),
        Gra::Gen => setup(target_dir),
        // Gra::Release => release(target_dir),
        Gra::InstallGSettings => {
            if !Uid::effective().is_root() {
                println!("You must run install-gsettings with root permissions.");
                return;
            }

            let gra_gen = target_dir.join(GEN_DIR);
            if let Ok(descriptor) = parse_project_descriptor(Path::new("Cargo.toml")) {
                let o = Command::new("install")
                    .arg("-D")
                    .arg(format!(
                        "{}/{}.gschema.xml",
                        gra_gen.to_str().unwrap(),
                        &descriptor.app.id
                    ))
                    .arg(GSETTINGS_SCHEMAS_DIR)
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    if !a.trim().is_empty() {
                        eprintln!("[gra] {}", a);
                        return;
                    }
                }
                let o = Command::new("glib-compile-schemas")
                    .arg(GSETTINGS_SCHEMAS_DIR)
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    if !a.trim().is_empty() {
                        eprintln!("[gra] {}", a);
                        return;
                    }
                }
                println!("[gra] Installed gsettings in {}.", GSETTINGS_SCHEMAS_DIR);
            } else {
                println!("[gra] Could not parse project descriptor.");
            }
        }
        Gra::UninstallGSettings => {
            if let Ok(descriptor) = parse_project_descriptor(Path::new("Cargo.toml")) {
                let o = Command::new("rm")
                    .arg(format!(
                        "{}/{}.gschema.xml",
                        GSETTINGS_SCHEMAS_DIR, &descriptor.app.id
                    ))
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    eprintln!("[gra] {}", a);
                }
                let o = Command::new("glib-compile-schemas")
                    .arg(GSETTINGS_SCHEMAS_DIR)
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    eprintln!("[gra] {}", a);
                }
            } else {
                println!("[gra] Could not parse project descriptor.");
            }
        }
        Gra::Flatpak(command) => flatpak(target_dir, command),
    }
}

fn clean(target_dir: PathBuf) {
    let flatpak_temp = target_dir.join("flatpak-temp");
    if flatpak_temp.exists() {
        remove_dir_all(&flatpak_temp).expect("Could not clean flatpak-temp");
    }
    let flatpak_build = target_dir.join("flatpak-build");
    if flatpak_build.exists() {
        remove_dir_all(&flatpak_build).expect("Could not clean flatpak-build");
    }
    let gra_gen = target_dir.join("gra-gen");
    if gra_gen.exists() {
        remove_dir_all(&gra_gen).expect("Could not clean gra-gen");
    }
}

fn flatpak(target_dir: PathBuf, arguments: Flatpak) {
    println!("[gra] Prepare flatpak build...");
    let pd = parse_project_descriptor(Path::new("Cargo.toml"));
    if pd.is_err() {
        eprintln!("[gra] Could not parse Cargo.toml: {}", pd.unwrap_err());
        return;
    }
    let project_descriptor = pd.unwrap();
    let gra_gen = target_dir.join("gra-gen");

    if arguments.release.is_some() {
        clean(target_dir.clone());
        setup(target_dir.clone());

        let flatpak_temp = match gtk_rust_app::prepare_flatpak_temp(&PathBuf::from(
            target_dir.parent().unwrap(),
        )) {
            Ok(f) => f,
            Err(e) => {
                eprintln!("Could not prepare flatpak-temp: {:?}", e);
                std::process::exit(-1);
            }
        };

        let manifest_yml = target_dir.join(format!("{}.yml", &project_descriptor.app.id));

        remove_file(gra_gen.join(format!("data/{}.dev.yml", &project_descriptor.app.id))).ok();
        if let Err(e) = std::fs::rename(
            gra_gen.join(format!("data/{}.yml", &project_descriptor.app.id)),
            &manifest_yml,
        ) {
            eprintln!(
                "[gra] Could not move target/gra-gen/data/{0}.yml to target/{0}.yml: {1}",
                &project_descriptor.app.id,
                e.to_string()
            );
            return;
        }

        let tar_file = target_dir.join(format!("{}.tar.xz", &project_descriptor.package.name));

        println!("[gra] Call tar -C {:?} -cJf {:?} .", flatpak_temp, tar_file);
        match Command::new("tar")
            .args([
                "-C",
                flatpak_temp.to_str().unwrap(),
                "-cJf",
                tar_file.to_str().unwrap(),
                ".",
            ])
            .spawn()
        {
            Err(e) => {
                eprintln!(
                    "[gra] tar flatpak-temp resulted in error: {}",
                    e.to_string()
                );
                return;
            }
            Ok(mut c) => {
                if let Err(e) = c.wait() {
                    eprintln!("[gra] tar command failed: {}", e.to_string());
                    return;
                }
            }
        }
        println!("[gra] Call sha256sum {:?}", &tar_file);

        let sha = Command::new("sha256sum").arg(&tar_file).output();
        if let Err(e) = sha {
            eprintln!(
                "[gra] sha256sum for {:?} resulted in error: {}",
                &tar_file,
                e.to_string()
            );
            return;
        }

        let sha = String::from_utf8_lossy(&sha.unwrap().stdout)
            .to_string()
            .split_once(" ")
            .unwrap()
            .0
            .trim()
            .to_string();

        let manifest_template = read_to_string(&manifest_yml).unwrap();
        let manifest = manifest_template
            .replace("{archive}", &arguments.release.unwrap())
            .replace("{sha256}", &sha);

        let sha_file = target_dir.join(format!("{}.sha256.txt", &project_descriptor.package.name));

        if let Ok(mut f) = File::create(&sha_file) {
            f.write_all(sha.as_bytes()).unwrap();
        } else {
            eprintln!("[gra] Could not create {:?}", sha_file);
            return;
        }

        if let Ok(mut f) = File::create(&manifest_yml) {
            f.write_all(manifest.as_bytes()).unwrap();
        } else {
            eprintln!("[gra] Could not write to release {:?}", manifest_yml);
            return;
        }

        println!(
            "[gra] Created flatpak release file: {:?}, {}",
            tar_file,
            manifest_yml.to_string_lossy()
        );

        return;
    }

    let flatpak_temp =
        match gtk_rust_app::prepare_flatpak_temp(&PathBuf::from(target_dir.parent().unwrap())) {
            Ok(f) => f,
            Err(e) => {
                eprintln!("Could not prepare flatpak-temp: {:?}", e);
                std::process::exit(-1);
            }
        };

    if arguments.prepare {
        return;
    }

    // setup flatpak-build dir
    let flatpak_build_rel = "../flatpak-build";

    println!("[gra] Running flatpak-builder --repo={0}/repo {0}/host {1}/data/{2}.dev.yml --force-clean --state-dir={0}/state {3}", 
        flatpak_build_rel,
        gra_gen.to_str().unwrap(),
        &project_descriptor.app.id,
        arguments.arch.as_ref().map(|a| format!("--arch {}", a)).unwrap_or("".into())
    );

    let mut c = Command::new("flatpak-builder");
    c.current_dir(&flatpak_temp);
    c.arg(format!("--repo={}/{}", flatpak_build_rel, "repo"));
    c.arg("--force-clean");
    c.arg(format!("--state-dir={}/{}", flatpak_build_rel, "state"));
    if let Some(arch) = &arguments.arch.as_ref() {
        c.arg("--arch").arg(arch);
    }
    if let Some(arch) = arguments.arch.as_ref() {
        c.arg(format!("{}/{}", flatpak_build_rel, arch));
    } else {
        c.arg(format!("{}/{}", flatpak_build_rel, "host"));
    }
    c.arg(
        gra_gen
            .join("data")
            .join(format!("{}.dev.yml", &project_descriptor.app.id)),
    );
    let mut c = c.spawn().unwrap();
    c.wait().unwrap();

    println!(
        "[gra] Bundling flatpak build-bundle {0}/repo {1}/{2}.flatpak {3}",
        flatpak_build_rel,
        target_dir.to_str().unwrap(),
        &project_descriptor.package.name,
        &project_descriptor.app.id
    );

    let flatpak_file_name = format!("{}.flatpak", &project_descriptor.package.name);

    let mut c = Command::new("flatpak")
        .current_dir(&flatpak_temp)
        .arg("build-bundle")
        .arg(format!("{}/repo", flatpak_build_rel))
        .arg(format!("../{}", flatpak_file_name))
        .arg(&project_descriptor.app.id)
        .spawn()
        .unwrap();
    c.wait().unwrap();

    if flatpak_temp.exists() {
        remove_dir_all(&flatpak_temp).expect("Could not clean flatpak-temp");
    }
}

fn setup(target_dir: PathBuf) {
    let gra_gen = target_dir.join("gra-gen");

    create_dir_all(&gra_gen).expect("Could not create target/gra-gen dir");
    gtk_rust_app::build(Some(gra_gen.as_path()));
}

// fn release(_target_dir: PathBuf) {
// let project_descriptor = parse_project_descriptor(Path::new("Cargo.toml")).unwrap();
// let c = Command::new("git")
//     .arg("diff-index")
//     .arg("--quiet")
//     .arg("HEAD")
//     .arg("--")
//     .output()
//     .unwrap();
// println!("{:?} {:?}", String::from_utf8(c.stdout));
// }
