// SPDX-License-Identifier: GPL-3.0-or-later

use clap::{Parser, Subcommand};
use fs_extra::dir::CopyOptions;
use gtk_rust_app::parse_project_descriptor;
use nix::unistd::Uid;
use std::{
    fs::{create_dir_all, remove_dir_all, File},
    io::Write,
    path::{Path, PathBuf},
    process::Command,
};

#[derive(Parser)]
#[clap(name = "cargo")]
#[clap(bin_name = "cargo")]
#[clap(override_usage = "cargo gra <SUBCOMMAND>")]
struct Cargo {
    // take the self name (gra) argument provided by cargo when calling via cargo gra ...
    #[clap(help = "Own subcommand name (gra)")]
    _self: Option<String>,
    #[clap(subcommand)]
    gra: Gra,
}

#[derive(Subcommand)]
#[clap(author, version, about)]
enum Gra {
    #[clap(about = "Cleanup generated files")]
    Clean,
    #[clap(
        about = "Generate manifests and source files from your Cargo.toml",
        override_usage = "cargo gra generate"
    )]
    Generate,
    #[clap(
        about = "Alias for cargo gra generate",
        override_usage = "cargo gra gen"
    )]
    Gen,
    #[clap(
        name = "install-gsettings",
        alias = "igs",
        about = "Install your apps gsettings schema (Requires sudo).",
        long_about = "Does the same as `install -D target/gra-gen/<app-id>.gschema.xml /usr/share/glib-2.0/schemas && glib-compile-schemas /usr/share/glib-2.0/schemas`",
        override_usage = "cargo gra install-gsettings"
    )]
    InstallGSettings,
    #[clap(
        name = "install-gsettings",
        alias = "ugs",
        about = "Uninstall your apps gsettings schema",
        long_about = "Does the same as `rm /usr/share/glib-2.0/schemas/<app-id>.gschema.xml && glib-compile-schemas /usr/share/glib-2.0/schemas`",
        override_usage = "cargo gra uninstall-gsettings"
    )]
    UninstallGSettings,
    #[clap(
        name = "flatpak",
        about = "Build a flatpak app locally",
        override_usage = "cargo gra flatpak"
    )]
    Flatpak(Flatpak),
}

#[derive(Parser)]
struct Flatpak {
    #[clap(
        short,
        long,
        help = "Only prepare the flatpak-temp. May be used to get everything set up and use flatpak-builder on your own."
    )]
    prepare: bool,
    #[clap(
        short,
        long,
        help = "The architecture to build for. The systems architecture will be used if unset. See https://docs.flatpak.org/en/latest/flatpak-builder-command-reference.html."
    )]
    arch: Option<String>,
}

const GEN_DIR: &str = "gra-gen";
const GSETTINGS_SCHEMAS_DIR: &str = "/usr/share/glib-2.0/schemas";

fn main() {
    let c: Cargo = Cargo::parse();

    let target_env = std::env::var("CARGO_TARGET_DIR").unwrap_or("target".into());
    let target_dir = PathBuf::new().join(target_env);

    match c.gra {
        Gra::Clean => clean(target_dir),
        Gra::Generate => setup(target_dir),
        Gra::Gen => setup(target_dir),
        // Gra::Release => release(target_dir),
        Gra::InstallGSettings => {
            if !Uid::effective().is_root() {
                println!("You must run install-gsettings with root permissions.");
                return;
            }

            let gra_gen = target_dir.join(GEN_DIR);
            if let Ok(descriptor) = parse_project_descriptor(Path::new("Cargo.toml")) {
                let o = Command::new("install")
                    .arg("-D")
                    .arg(format!(
                        "{}/{}.gschema.xml",
                        gra_gen.to_str().unwrap(),
                        &descriptor.app.unwrap().id
                    ))
                    .arg(GSETTINGS_SCHEMAS_DIR)
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    if !a.trim().is_empty() {
                        eprintln!("[gra] {}", a);
                        return;
                    }
                }
                let o = Command::new("glib-compile-schemas")
                    .arg(GSETTINGS_SCHEMAS_DIR)
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    if !a.trim().is_empty() {
                        eprintln!("[gra] {}", a);
                        return;
                    }
                }
                println!("[gra] Installed gsettings in {}.", GSETTINGS_SCHEMAS_DIR);
            } else {
                println!("[gra] Could not parse project descriptor.");
            }
        }
        Gra::UninstallGSettings => {
            if let Ok(descriptor) = parse_project_descriptor(Path::new("Cargo.toml")) {
                let o = Command::new("rm")
                    .arg(format!(
                        "{}/{}.gschema.xml",
                        GSETTINGS_SCHEMAS_DIR,
                        &descriptor.app.unwrap().id
                    ))
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    eprintln!("[gra] {}", a);
                }
                let o = Command::new("glib-compile-schemas")
                    .arg(GSETTINGS_SCHEMAS_DIR)
                    .output()
                    .unwrap();
                if let Ok(a) = String::from_utf8(o.stderr) {
                    eprintln!("[gra] {}", a);
                }
            } else {
                println!("[gra] Could not parse project descriptor.");
            }
        }
        Gra::Flatpak(command) => flatpak(target_dir, command),
    }
}

fn clean(target_dir: PathBuf) {
    let flatpak_temp = target_dir.join("flatpak-temp");
    if flatpak_temp.exists() {
        remove_dir_all(&flatpak_temp).expect("Could not clean flatpak-temp");
    }
    let flatpak_build = target_dir.join("flatpak-build");
    if flatpak_build.exists() {
        remove_dir_all(&flatpak_build).expect("Could not clean flatpak-build");
    }
    let gra_gen = target_dir.join("gra-gen");
    if gra_gen.exists() {
        remove_dir_all(&gra_gen).expect("Could not clean gra-gen");
    }
}

fn flatpak(target_dir: PathBuf, arguments: Flatpak) {
    println!("[gra] Prepare flatpak build...");
    let project_descriptor = parse_project_descriptor(Path::new("Cargo.toml")).unwrap();

    let gra_gen = target_dir.join("gra-gen");

    // setup flatpak-temp dir
    let flatpak_temp = target_dir.join("flatpak-temp");
    if flatpak_temp.exists() {
        remove_dir_all(&flatpak_temp).expect("Could not clean flatpak-temp");
    }

    println!("mkdir target/flatpak-temp");
    create_dir_all(&flatpak_temp).expect("Could not create flatpak-temp");
    println!("[gra] mkdir target/flatpak-temp/target");
    create_dir_all(&flatpak_temp.join("target")).expect("Could not create flatpak-temp/target");
    println!("[gra] mkdir target/flatpak-temp/.cargo");
    create_dir_all(&flatpak_temp.join(".cargo")).expect("Could not create flatpak-temp/.cargo");

    let mut options = CopyOptions::new();
    options.overwrite = true;
    options.copy_inside = true;

    println!("[gra] cp -r src target/flatpak-temp");
    fs_extra::dir::copy("src", &flatpak_temp, &options)
        .expect("Copy src dir to target/flatpak-temp");
    println!("[gra] cp -r po target/flatpak-temp");
    fs_extra::dir::copy("po", &flatpak_temp, &options).expect("Copy po dir to target/flatpak-temp");
    println!("[gra] cp Cargo.toml target/flatpak-temp");
    std::fs::copy("Cargo.toml", &flatpak_temp.join("Cargo.toml"))
        .expect("Copy Cargo.toml to target/flatpak-temp");
    println!("[gra] cp -r target/gra-gen target/flatpak-temp/target");
    fs_extra::dir::copy("target/gra-gen", &flatpak_temp.join("target"), &options)
        .expect("Copy target/gra-gen to target/flatpak-temp/target/");

    println!("[gra] Vendoring sources...");
    let c = Command::new("cargo")
        .current_dir(&flatpak_temp)
        .args(["vendor", "target/vendor"])
        .output()
        .unwrap();

    if let Ok(e) = String::from_utf8(c.stderr) {
        if !e.trim().is_empty() {
            println!("[gra] {}", e);
        }
    }
    let mut config = File::create(flatpak_temp.join(".cargo").join("config.toml")).unwrap();
    config.write_all(&c.stdout).unwrap();

    if arguments.prepare {
        return;
    }

    // setup flatpak-build dir
    let flatpak_build = target_dir.join("flatpak-build");

    println!("[gra] Running flatpak-builder --repo={0}/repo {0}/host {1}/data/{2}.yml --force-clean --state-dir={0}/state", 
        flatpak_build.to_str().unwrap(),
        gra_gen.to_str().unwrap(),
        &project_descriptor.app.as_ref().unwrap().id);

    let mut c = Command::new("flatpak-builder");
    c.arg(format!(
        "--repo={}",
        flatpak_build.join("repo").to_str().unwrap()
    ));

    if let Some(arch) = arguments.arch.as_ref() {
        c.arg(flatpak_build.join(arch));
    } else {
        c.arg(flatpak_build.join("host"));
    }

    c.arg(gra_gen.join("data").join(format!(
        "{}.yml",
        project_descriptor.app.as_ref().unwrap().id
    )));
    c.arg("--force-clean");
    c.arg(format!(
        "--state-dir={}",
        flatpak_build.join("state").to_str().unwrap()
    ));
    if let Some(arch) = &arguments.arch.as_ref() {
        c.arg("--arch").arg(arch);
    }
    let mut c = c.spawn().unwrap();
    c.wait().unwrap();

    println!(
        "[gra] Bundling {}/{}.flatpak",
        target_dir.to_str().unwrap(),
        &project_descriptor.package.name
    );
    let mut c = Command::new("flatpak")
        .arg("build-bundle")
        .arg(flatpak_build.join("repo"))
        .arg(target_dir.join(format!("{}.flatpak", &project_descriptor.package.name)))
        .arg(&project_descriptor.app.as_ref().unwrap().id)
        .spawn()
        .unwrap();
    c.wait().unwrap();

    if flatpak_temp.exists() {
        remove_dir_all(&flatpak_temp).expect("Could not clean flatpak-temp");
    }
}

fn setup(target_dir: PathBuf) {
    let gra_gen = target_dir.join("gra-gen");

    create_dir_all(&gra_gen).expect("Could not create target/gra-gen dir");
    gtk_rust_app::build(Some(gra_gen.as_path()));
}

// fn release(_target_dir: PathBuf) {
// let project_descriptor = parse_project_descriptor(Path::new("Cargo.toml")).unwrap();
// let c = Command::new("git")
//     .arg("diff-index")
//     .arg("--quiet")
//     .arg("HEAD")
//     .arg("--")
//     .output()
//     .unwrap();
// println!("{:?} {:?}", String::from_utf8(c.stdout));
// }
