use std::collections::{HashMap, HashSet};
use std::path::{Path, PathBuf};

use anyhow::{bail, Context};
use cargo_util::{paths, ProcessBuilder};
use faccess::PathExt;
use tracing::debug;

use crate::args::BuildArgs;
use crate::config::Config;
use crate::error::CIError;
use crate::{cargo, util, CIResult};

pub fn exec(config: Config, args: BuildArgs) -> CIResult<()> {
    if !Path::new(&config.library_path).is_file() {
        bail!(CIError::LibraryNotInstalled);
    }

    let mut llvm_bins = ["opt", "llc", "llvm-ar", "llvm-nm"]
        .iter()
        .map(|&s| s.to_string())
        .collect();
    util::llvm_toolchain(&mut llvm_bins)?;

    println!("Compiling...");

    let mut mtimes = HashMap::new();
    let mut stale_files = Vec::new();

    let target_dir = util::target_dir(&args.target, &args.release)?;
    let deps_dir = target_dir.join("deps");
    let deps_ci_dir = target_dir.join("deps-ci");

    // create new "deps-ci" directory for CI-integrated files
    paths::create_dir_all(&deps_ci_dir)?;

    // get timestamp from output files before running `cargo build`
    if let Ok(target_files) = util::scan_dir(&deps_dir, |path| path.is_file()) {
        for file in target_files {
            let mtime = paths::mtime(&file)?;
            assert!(mtimes.insert(file, mtime).is_none());
        }
    }

    // run `cargo build`
    let cargo_build = cargo::build(&args)?;
    debug!("cargo_build: {:?}", cargo_build);

    // check for stale files after the compilation
    let deps_files = util::scan_dir(&deps_dir, |path| path.is_file())?;
    for file in &deps_files {
        let new_mtime = paths::mtime(&file)?;
        match mtimes.get(file) {
            Some(cache_mtime) => {
                if new_mtime > *cache_mtime {
                    stale_files.push(file);
                }
            }
            None => {
                mtimes.insert(file.clone(), new_mtime);
                stale_files.push(file);
            }
        }
    }

    debug!("stale_files: {:#?}", stale_files);

    if stale_files.is_empty() {
        println!("Nothing to integrate, all fresh");
        return Ok(());
    }

    println!("Integrating...");

    // get binary names
    let mut bin_names = vec![];
    let metadata = cargo::metadata()?;
    for package in metadata.packages {
        for target in package.targets {
            let t = target.crate_types.iter();
            let k = target.kind.iter();
            if t.zip(k).all(|(t, k)| t == "bin" && k == "bin") {
                bin_names.push(target.name.replace("-", "_"));
            }
        }
    }
    debug!("bin_names: {:#?}", bin_names);

    // name of stale binaries
    let stale_bin_names = stale_files
        .iter()
        .filter(|file| bin_names.contains(&binary_name(file)))
        .map(|file| binary_name(file))
        .collect::<HashSet<_>>();
    debug!("stale_bin_names: {:#?}", stale_bin_names);

    // *.rcgu.ll are intermediate files generated by `rustc -C save-temps`
    let ll_files = deps_files
        .iter()
        .filter(|path| {
            util::file_stem_unwrapped(path).contains("rcgu")
                && util::extension_unwrapped(path) == "ll"
        })
        .collect::<Vec<_>>();

    // run the integration
    let config = std::sync::Arc::new(config);
    let mut threads = vec![];
    for file in ll_files {
        let file = file.clone();
        let config = config.clone();
        let bin_names = bin_names.clone();
        let llvm_bins = llvm_bins.clone();

        let thread = std::thread::spawn(move || -> CIResult<()> {
            debug!("integrating: {}", file.display());
            let ci_file = util::append_suffix(r_depsci(&file), "ci");

            // `nm -jU` displays defined symbol names
            let nm = ProcessBuilder::new(&llvm_bins[3])
                .arg("-jU")
                .arg(file.with_extension("o"))
                .exec_with_output()?;
            let stdout = String::from_utf8(nm.stdout)?;
            if stdout.contains("_intvActionHook") {
                // skip object file contains CI symbols
                paths::copy(&file, &ci_file)?;
            } else {
                // define `LocalLC` if it is a binary target
                let def_clock = match bin_names.contains(&binary_name(&file)) {
                    true => "-defclock=1",
                    false => "-defclock=0",
                };

                let mut args = [
                    "-S",
                    "-load",
                    &config.library_path,
                    "-logicalclock",
                    def_clock,
                ]
                .iter()
                .map(|s| s.to_string())
                .collect::<Vec<_>>();
                args.extend(config.default_args.clone());

                // `opt` runs the integration
                ProcessBuilder::new(&llvm_bins[0])
                    .args(&args)
                    .arg(&file)
                    .arg("-o")
                    .arg(&ci_file)
                    .exec_with_output()?;
            }

            // `llc` transforms integrated IR bitcode to object file
            ProcessBuilder::new(&llvm_bins[1])
                .arg("-filetype=obj")
                .arg(ci_file)
                .exec_with_output()?;

            Ok(())
        });
        threads.push(thread);
    }
    for thread in threads {
        thread.join().expect("failed to join thread")?;
    }

    // sample of the expected `cargo build` output
    // INFO rustc_codegen_ssa::back::link preparing Executable to "/path/to/binary"
    // INFO rustc_codegen_ssa::back::link "cc" "-m64 "-arch" "x86_64" ...

    // parse output to get the linker command
    debug!("parsing linker");
    let stderr = String::from_utf8(cargo_build.stderr)?;
    let mut infos = stderr
        .lines()
        .filter(|x| x.contains("INFO rustc_codegen_ssa::back::link"));

    'outer: while let Some(info) = infos.next() {
        if !info.contains("libcompiler_builtins") {
            // ignore if this is not a linker command
            continue;
        }

        let mut cmd = info
            .replace("\"", "")
            .split_ascii_whitespace()
            .skip(2) // skip "INFO", "rustc_codegen_ssa::back::link"
            .map(str::to_string)
            .collect::<Vec<_>>();

        // parse the args to find the name of the binary
        let mut iter = cmd.iter();
        while let Some(arg) = iter.next() {
            if arg.contains("-o") {
                let bin_path = iter.next().context("expect path to binary")?;
                let bin_name = binary_name(bin_path);

                if !stale_bin_names.contains(&bin_name) {
                    // redundant linker cmd since the binary is still fresh
                    debug!("skipping linker invocation for binary \"{}\"", bin_name);
                    continue 'outer;
                }
            }
        }

        debug!("finding allocator shim");
        let object_files = cmd.iter_mut().filter(|e| e.contains(".o"));
        for file in object_files {
            // find the object file contains the symbol for memory allocator
            let nm = ProcessBuilder::new(&llvm_bins[3])
                .arg("-jU")
                .arg(&file)
                .exec_with_output()?;
            let stdout = String::from_utf8(nm.stdout)?;
            if stdout.contains("__rust_alloc") {
                debug!("found allocator shim: {}", file);
                paths::copy(&file, r_depsci(&file))?;
            } else {
                *file = util::append_suffix(&file, "ci").display().to_string();
            }
        }
        debug!("finding allocator shim... done");

        let deps_rlib_files = cmd
            .iter_mut()
            .filter(|e| e.contains("deps") && e.contains(".rlib"));
        for file in deps_rlib_files {
            debug!("replacing object file for rlib: {}", file);

            // list all object files inside rlib
            let ar = ProcessBuilder::new(&llvm_bins[2])
                .arg("-t")
                .arg(&file)
                .exec_with_output()?;
            let stdout = String::from_utf8(ar.stdout)?;
            if let Some(rcgu_obj_file_name) = stdout.lines().find(|e| e.contains("rcgu")) {
                debug!("found obj file: {}", rcgu_obj_file_name);
                let rcgu_obj_file = deps_dir.join(rcgu_obj_file_name);
                let rcgu_obj_ci_file = util::append_suffix(r_depsci(&rcgu_obj_file), "ci");

                // copy the rlib file to the "deps-ci" dir
                let ci_file = r_depsci(&file);
                paths::copy(&file, &ci_file)?;

                // replace *.o with *-ci.o
                ProcessBuilder::new(&llvm_bins[2])
                    .arg("-rb")
                    .arg(&rcgu_obj_file)
                    .arg(&ci_file)
                    .arg(&rcgu_obj_ci_file)
                    .exec_with_output()?;

                // delete old *.o 
                ProcessBuilder::new(&llvm_bins[2])
                    .arg("-d")
                    .arg(&ci_file)
                    .arg(&rcgu_obj_file)
                    .exec_with_output()?;
            }
        }

        // replace all "deps" to "deps-ci"
        for arg in &mut cmd {
            *arg = arg.replace("deps", "deps-ci");
        }

        // execute the linker
        debug!("executing linker");
        debug!("linker: {:#?}", cmd);
        let mut iter = cmd.iter();
        let mut linker = ProcessBuilder::new(iter.next().context("expect linker name")?);
        for arg in iter {
            linker.arg(arg);
        }
        let output = linker.exec_with_output()?;
        debug!("linker output: {:?}", output);
    }

    // copy CI-integrated binary file to the parent directory
    let deps_ci_files = util::scan_dir(&deps_ci_dir, |path| path.is_file())?;
    let binary_files = deps_ci_files
        .iter()
        .filter(|path| path.executable() && path.is_file())
        .collect::<Vec<_>>();
    for file in binary_files {
        let file_name = binary_name(&file).replace("_", "-");
        let parent = file.parent().context("failed to get parent dir")?;
        let path = parent.with_file_name(file_name);
        let path = util::append_suffix(&path, "ci");
        paths::copy(file, path)?;
    }

    println!("Done. Run `cargo run-ci` to execute the integrated binary");

    Ok(())
}

/// Get the binary name from path.
fn binary_name<P: AsRef<Path>>(path: P) -> String {
    util::file_stem_unwrapped(path)
        .split('.')
        .next()
        .unwrap()
        .split('-')
        .next()
        .unwrap()
        .to_string()
}

/// Replace "deps" with "deps-ci" for the path.
fn r_depsci<P: AsRef<Path>>(path: P) -> PathBuf {
    let path = path.as_ref();
    let file_name = util::file_name_unwrapped(path);
    let mut path = PathBuf::from(path);
    path.pop();
    path.set_file_name("deps-ci");
    path.push(file_name);
    path
}
