use {
    crate::{state::*, errors::*},
    anchor_lang::{{prelude::*}, AccountsClose}
};
use anchor_spl::{
    token::{self, Token, TokenAccount, Transfer, Mint, ThawAccount, CloseAccount},
};

#[derive(Accounts)]
pub struct RevokeCertificateCtx<'info> {
    #[account(seeds = [MINT_MANAGER_SEED.as_bytes(), certificate.mint.as_ref()], bump = mint_manager.bump)]
    pub mint_manager: Box<Account<'info, MintManager>>, 
    #[account(mut, constraint = certificate.state == CertificateState::Claimed as u8)]
    pub certificate: Box<Account<'info, Certificate>>,
    #[account(mut, constraint = certificate_mint.key() == certificate.mint)]
    pub certificate_mint: Box<Account<'info, Mint>>,
    #[account(mut, constraint = certificate_token_account.owner == certificate.key() @ ErrorCode::InvalidCertificateTokenAccount)]
    pub certificate_token_account: Box<Account<'info, TokenAccount>>,
    #[account(mut, constraint = certificate.payment_mint == None || certificate.payment_amount == None || (certificate_payment_token_account.owner == certificate.key() && certificate_payment_token_account.mint == certificate.payment_mint.unwrap()) @ ErrorCode::InvalidCertificatePaymentTokenAccount)]
    pub certificate_payment_token_account: Box<Account<'info, TokenAccount>>,

    // recipient
    #[account(mut)]
    pub recipient_token_account: Box<Account<'info, TokenAccount>>,
    #[account(mut, constraint = certificate.payment_mint == None || certificate.payment_amount == None || (recipient_payment_token_account.owner == certificate.recipient.unwrap() && recipient_payment_token_account.mint == certificate.payment_mint.unwrap()) @ ErrorCode::InvalidRecipientPaymentTokenAccount)]
    pub recipient_payment_token_account: Box<Account<'info, TokenAccount>>,

    // issuer
    #[account(mut, constraint = issuer_token_account.owner == certificate.issuer @ ErrorCode::InvalidOwnership)]
    pub issuer_token_account: Box<Account<'info, TokenAccount>>,
    #[account(mut, constraint = certificate.payment_mint == None || certificate.payment_amount == None || (issuer_payment_token_account.owner == certificate.issuer && issuer_payment_token_account.mint == certificate.payment_mint.unwrap()) @ ErrorCode::InvalidRecipientPaymentTokenAccount)]
    pub issuer_payment_token_account: Box<Account<'info, TokenAccount>>,
    
    // other
    #[account(mut,
        constraint = certificate.is_returnable && revoke_authority.key() == certificate.recipient.unwrap()
        || certificate.revoke_authority != None && revoke_authority.key() == certificate.revoke_authority.unwrap() @ ErrorCode::CannotRevoke,
    )]
    pub revoke_authority: Signer<'info>,
    pub token_program: Program<'info, Token>,
}

pub fn handler(ctx: Context<RevokeCertificateCtx>) -> ProgramResult {
    let certificate = &ctx.accounts.certificate;
    let mint_manager_seeds = &[MINT_MANAGER_SEED.as_bytes(), certificate.mint.as_ref(), &[ctx.accounts.mint_manager.bump]];
    let mint_manager_signer = &[&mint_manager_seeds[..]];
    let certificate_seeds = &[CERTIFICATE_SEED.as_bytes(), certificate.mint.as_ref(), &[certificate.bump]];
    let certificate_signer = &[&certificate_seeds[..]];

    // transfer prices back to relevent parties
    if certificate.payment_amount != None && certificate.payment_amount.unwrap() > 0 {
        let mut used_pct = 0 as f64;
        if certificate.max_usages != None {
            used_pct = used_pct.max(certificate.usages as f64 / certificate.max_usages.unwrap() as f64)
        }
        if certificate.expiration != None {
            used_pct = used_pct.max((Clock::get().unwrap().unix_timestamp - certificate.issued_at) as f64 / (certificate.expiration.unwrap() - certificate.issued_at) as f64)
        }
        let used_amount = (certificate.payment_amount.unwrap() as f64 * used_pct) as u64;

        // transfer max of used percentage to the issuer
        let cpi_accounts = Transfer {
            from: ctx.accounts.certificate_payment_token_account.to_account_info(),
            to: ctx.accounts.issuer_payment_token_account.to_account_info(),
            authority: certificate.to_account_info(),
        };
        let cpi_program = ctx.accounts.token_program.to_account_info();
        let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
        token::transfer(cpi_context, used_amount)?;

        // transfer rest back to recipient
        let cpi_accounts = Transfer {
            from: ctx.accounts.certificate_payment_token_account.to_account_info(),
            to: ctx.accounts.recipient_payment_token_account.to_account_info(),
            authority: certificate.to_account_info(),
        };
        let cpi_program = ctx.accounts.token_program.to_account_info();
        let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
        token::transfer(cpi_context, certificate.payment_amount.unwrap() - used_amount)?;

        // close certificate payment token account
        let cpi_accounts = CloseAccount {
            account: ctx.accounts.certificate_payment_token_account.to_account_info(),
            destination: ctx.accounts.revoke_authority.to_account_info(),
            authority: certificate.to_account_info(),
        };
        let cpi_program = ctx.accounts.token_program.to_account_info();
        let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
        token::close_account(cpi_context)?;
    }

    if certificate.kind == CertificateKind::Managed as u8 {
        // thaw recipient account
        let cpi_accounts = ThawAccount {
            account: ctx.accounts.recipient_token_account.to_account_info(),
            mint: ctx.accounts.certificate_mint.to_account_info(),
            authority: ctx.accounts.mint_manager.to_account_info(),
        };
        let cpi_program = ctx.accounts.token_program.to_account_info();
        let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(mint_manager_signer);
        token::thaw_account(cpi_context)?;

        // transfer amount to issuer
        let cpi_accounts = Transfer {
            from: ctx.accounts.recipient_token_account.to_account_info(),
            to: ctx.accounts.issuer_token_account.to_account_info(),
            authority: certificate.to_account_info(),
        };
        let cpi_program = ctx.accounts.token_program.to_account_info();
        let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
        token::transfer(cpi_context, certificate.amount)?;
        
        // close certificate account
        ctx.accounts.certificate.close(ctx.accounts.revoke_authority.to_account_info())?;
    }

    // close certificate token account
    let cpi_accounts = CloseAccount {
        account: ctx.accounts.certificate_token_account.to_account_info(),
        destination: ctx.accounts.revoke_authority.to_account_info(),
        authority: certificate.to_account_info(),
    };
    let cpi_program = ctx.accounts.token_program.to_account_info();
    let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
    token::close_account(cpi_context)?;

    // decrement mint_manager
    let mint_manager = &mut ctx.accounts.mint_manager;
    mint_manager.outstanding_certificates -= 1;
    return Ok(())
}