use {
    crate::{state::*, errors::*},
    anchor_lang::{prelude::*}
};
use anchor_spl::{
    token::{self, Token, TokenAccount, Transfer},
};

#[derive(Accounts)]
pub struct ExtendCertificateCtx<'info> {
    #[account(mut, constraint = certificate.is_extendable @ ErrorCode::CannotExtend)]
    pub certificate: Account<'info, Certificate>,
    #[account(mut, constraint = certificate.payment_mint == None || (certificate_payment_token_account.owner == certificate.key() && certificate_payment_token_account.mint == certificate.payment_mint.unwrap()) @ ErrorCode::InvalidCertificatePaymentTokenAccount)]
    pub certificate_payment_token_account: Box<Account<'info, TokenAccount>>,

    pub payer: Signer<'info>,
    #[account(mut, constraint = payment_token_account.owner == payer.key() @ ErrorCode::InvalidIssuerPaymentTokenAccount)]
    pub payment_token_account: Account<'info, TokenAccount>,
    pub token_program: Program<'info, Token>,
}

pub fn handler(ctx: Context<ExtendCertificateCtx>, amount: u64) -> ProgramResult {
    let certificate = &mut ctx.accounts.certificate;
    // extend expiration
    if certificate.expiration != None {
        let price_per_second = certificate.payment_amount.unwrap() as f64 / certificate.duration.unwrap() as f64;
        certificate.expiration = Some(certificate.expiration.unwrap() + ((amount as f64 / price_per_second).round() as i64));
    }
    // extend usages
    if certificate.max_usages != None {
        let price_per_usage = certificate.payment_amount.unwrap() as f64 / certificate.total_usages.unwrap() as f64;
        certificate.max_usages = Some(certificate.max_usages.unwrap() + ((amount as f64 / price_per_usage).round() as u64));
    }
    // add funds to certificate
    let cpi_accounts = Transfer {
        from: ctx.accounts.payment_token_account.to_account_info(),
        to: ctx.accounts.certificate_payment_token_account.to_account_info(),
        authority: ctx.accounts.payer.to_account_info(),
    };
    let cpi_program = ctx.accounts.token_program.to_account_info();
    let cpi_context = CpiContext::new(cpi_program, cpi_accounts);
    token::transfer(cpi_context, certificate.payment_amount.unwrap())?;
    return Ok(())
}