use {
    crate::{state::*, errors::*},
    anchor_lang::{prelude::*}
};
use anchor_spl::{
    token::{self, Token, Mint, TokenAccount, Transfer, Approve, FreezeAccount},
};

#[derive(Accounts)]
pub struct ClaimCertificateCtx<'info> {
    #[account(seeds = [MINT_MANAGER_SEED.as_bytes(), certificate.mint.as_ref()], bump = mint_manager.bump)]
    pub mint_manager: Box<Account<'info, MintManager>>,
    #[account(mut, constraint = certificate.state == CertificateState::Issued as u8 @ ErrorCode::CertificateNotIssued)]
    pub certificate: Box<Account<'info, Certificate>>,
    #[account(constraint = certificate_mint.key() == certificate.mint @ ErrorCode::InvalidCertificateMint)]
    pub certificate_mint: Box<Account<'info, Mint>>,
    #[account(mut, constraint = certificate_token_account.owner == certificate.key() && certificate_token_account.mint == certificate.mint @ ErrorCode::InvalidCertificateTokenAccount)]
    pub certificate_token_account: Box<Account<'info, TokenAccount>>,
    #[account(mut, constraint = certificate.payment_mint == None || (certificate_payment_token_account.owner == certificate.key() && certificate_payment_token_account.mint == certificate.payment_mint.unwrap()) @ ErrorCode::InvalidCertificatePaymentTokenAccount)]
    pub certificate_payment_token_account: Box<Account<'info, TokenAccount>>,

    // recipient
    #[account(mut, constraint = certificate.recipient == None || certificate.recipient.unwrap() == recipient.key() @ ErrorCode::UnauthorizedRecipient)]
    pub recipient: Signer<'info>,
    #[account(mut, constraint = recipient_token_account.owner == recipient.key() && recipient_token_account.mint == certificate.mint @ ErrorCode::InvalidRecipientTokenAccount)]
    pub recipient_token_account: Box<Account<'info, TokenAccount>>,
    #[account(mut, constraint = certificate.payment_mint == None || (recipient_payment_token_account.owner == recipient.key() && recipient_payment_token_account.mint == certificate.payment_mint.unwrap()) @ ErrorCode::InvalidRecipientPaymentTokenAccount)]
    pub recipient_payment_token_account: Box<Account<'info, TokenAccount>>,

    // other
    pub token_program: Program<'info, Token>,
}

pub fn handler(ctx: Context<ClaimCertificateCtx>) -> ProgramResult {
    let certificate = &mut ctx.accounts.certificate;
    certificate.recipient = Some(ctx.accounts.recipient.key());
    certificate.recipient_token_account = Some(ctx.accounts.recipient_token_account.key());
    certificate.claimed_at = Clock::get().unwrap().unix_timestamp;
    certificate.state = CertificateState::Claimed as u8;

    // set expiration of claimed time + duration if duration set
    if certificate.duration != None && certificate.start_at_issuance != None && !certificate.start_at_issuance.unwrap() {
        certificate.expiration = Some(certificate.claimed_at + certificate.duration.unwrap());
    }

    // get PDA seeds to sign with
    let mint_manager_seeds = &[MINT_MANAGER_SEED.as_bytes(), certificate.mint.as_ref(), &[ctx.accounts.mint_manager.bump]];
    let mint_manager_signer = &[&mint_manager_seeds[..]];
    let certificate_seeds = &[CERTIFICATE_SEED.as_bytes(), certificate.mint.as_ref(), &[certificate.bump]];
    let certificate_signer = &[&certificate_seeds[..]];

    // transfer price into certificate account
    if certificate.payment_amount != None && certificate.payment_amount.unwrap() > 0 {
        // transfer amount to payment token account
        let cpi_accounts = Transfer {
            from: ctx.accounts.recipient_payment_token_account.to_account_info(),
            to: ctx.accounts.certificate_payment_token_account.to_account_info(),
            authority: ctx.accounts.recipient.to_account_info(),
        };
        let cpi_program = ctx.accounts.token_program.to_account_info();
        let cpi_context = CpiContext::new(cpi_program, cpi_accounts);
        token::transfer(cpi_context, certificate.payment_amount.unwrap())?;
    }
    
    // transfer amount to recipient token account
    let cpi_accounts = Transfer {
        from: ctx.accounts.certificate_token_account.to_account_info(),
        to: ctx.accounts.recipient_token_account.to_account_info(),
        authority: certificate.to_account_info(),
    };
    let cpi_program = ctx.accounts.token_program.to_account_info();
    let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
    token::transfer(cpi_context, certificate.amount)?;

    // if this is a managed certificate, this means we will revoke it at the end of life, so we need to delegate and freeze
    if certificate.kind == CertificateKind::Managed as u8 {
        // set account delegate of recipient token account to certificate PDA
        let cpi_accounts = Approve {
            to: ctx.accounts.recipient_token_account.to_account_info(),
            delegate: certificate.to_account_info(),
            authority: ctx.accounts.recipient.to_account_info(),
        };
        let cpi_program = ctx.accounts.token_program.to_account_info();
        let cpi_context = CpiContext::new(cpi_program, cpi_accounts);
        token::approve(cpi_context, certificate.amount)?;
        
        // freeze recipient token account
        let cpi_accounts = FreezeAccount {
            account: ctx.accounts.recipient_token_account.to_account_info(),
            mint: ctx.accounts.certificate_mint.to_account_info(),
            authority: ctx.accounts.mint_manager.to_account_info(),
        };
        let cpi_program = ctx.accounts.token_program.to_account_info();
        let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(mint_manager_signer);
        token::freeze_account(cpi_context)?;
    }
    return Ok(())
}