use {
    crate::{state::*, errors::*},
    anchor_lang::{prelude::*}
};
use anchor_lang::AccountsClose;
use anchor_spl::{
    token::{self, Token, Mint, TokenAccount, Transfer, CloseAccount},
};

#[derive(Accounts)]
pub struct UnissueCertificateCtx<'info> {
    #[account(seeds = [MINT_MANAGER_SEED.as_bytes(), certificate.mint.as_ref()], bump = mint_manager.bump)]
    pub mint_manager: Account<'info, MintManager>, 
    #[account(
        mut,
        seeds = [CERTIFICATE_SEED.as_bytes(), certificate.mint.as_ref()], bump = certificate.bump,
        constraint = certificate.state == CertificateState::Issued as u8
    )]
    pub certificate: Account<'info, Certificate>,
    #[account(mut, constraint = certificate_token_account.owner == certificate.key() @ ErrorCode::InvalidCertificateTokenAccount)]
    pub certificate_token_account: Box<Account<'info, TokenAccount>>,
    #[account(mut, constraint = certificate.mint == certificate_mint.key())]
    pub certificate_mint: Box<Account<'info, Mint>>,

    // issuer
    #[account(mut, constraint = certificate.issuer == issuer.key() @ ErrorCode::InvalidIssuer)]
    pub issuer: Signer<'info>,
    #[account(mut, constraint = issuer_token_account.owner == issuer.key() @ ErrorCode::InvalidOwnership)]
    pub issuer_token_account: Box<Account<'info, TokenAccount>>,

    // other
    pub token_program: Program<'info, Token>,
}

pub fn handler(ctx: Context<UnissueCertificateCtx>) -> ProgramResult {
    let certificate = &ctx.accounts.certificate;
        
    // get PDA seeds to sign with
    let certificate_seeds = &[CERTIFICATE_SEED.as_bytes(), certificate.mint.as_ref(), &[certificate.bump]];
    let certificate_signer = &[&certificate_seeds[..]];

    // transfer amount to destination token account
    let cpi_accounts = Transfer {
        from: ctx.accounts.certificate_token_account.to_account_info(),
        to: ctx.accounts.issuer_token_account.to_account_info(),
        authority: ctx.accounts.certificate.to_account_info(),
    };
    let cpi_program = ctx.accounts.token_program.to_account_info();
    let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
    token::transfer(cpi_context, certificate.amount)?;

    // close certificate account
    ctx.accounts.certificate.close(ctx.accounts.issuer.to_account_info())?;

    // close token account
    let cpi_accounts = CloseAccount {
        account: ctx.accounts.certificate_token_account.to_account_info(),
        destination: ctx.accounts.issuer.to_account_info(),
        authority: certificate.to_account_info(),
    };
    let cpi_program = ctx.accounts.token_program.to_account_info();
    let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
    token::close_account(cpi_context)?;

    // decrement mint_manager
    let mint_manager = &mut ctx.accounts.mint_manager;
    mint_manager.outstanding_certificates -= 1;
    return Ok(())
}