use {
    crate::{state::*, errors::*},
    anchor_lang::{prelude::*}
};
use anchor_spl::{
    token::{self, Token, Mint, TokenAccount, Transfer},
};

#[derive(AnchorSerialize, AnchorDeserialize)]
pub struct IssueCertificateIx {
    // basic
    pub amount: u64,
    pub kind: u8,
    pub bump: u8,
    pub recipient: Option<Pubkey>,

    // pay
    pub payment_amount: Option<u64>,
    pub payment_mint: Option<Pubkey>,

    // time expiry
    pub duration: Option<i64>,
    pub start_at_issuance: Option<bool>,

    // usage expiry
    pub total_usages: Option<u64>,

    // manual expiry
    pub revoke_authority: Option<Pubkey>,

    // invalidation
    pub is_returnable: bool,
    pub is_extendable: bool,
}


#[derive(Accounts)]
#[instruction(ix: IssueCertificateIx)]
pub struct IssueCertificateCtx<'info> {
    #[account(
        constraint = ix.kind == CertificateKind::Unmanaged as u8 || certificate_mint.freeze_authority.unwrap() == mint_manager.key() @ ErrorCode::InvalidFreezeAuthority,
        seeds = [MINT_MANAGER_SEED.as_bytes(), certificate_mint.key().as_ref()],
        bump = mint_manager.bump
    )]
    mint_manager: Box<Account<'info, MintManager>>,
    #[account(
        init,
        payer = payer,
        space = CERTIFICATE_SIZE,
        seeds = [CERTIFICATE_SEED.as_bytes(), certificate_mint.key().as_ref()], bump = ix.bump,
    )]
    certificate: Box<Account<'info, Certificate>>,
    certificate_mint: Box<Account<'info, Mint>>,
    #[account(mut, constraint = certificate_token_account.owner == certificate.key() @ ErrorCode::CertificateMustOwnTokenAccount)]
    certificate_token_account: Box<Account<'info, TokenAccount>>,

    // issuer
    issuer: Signer<'info>,
    #[account(mut, constraint = issuer_token_account.owner == issuer.key() @ ErrorCode::IssuerMustOwnTokenAccount)]
    issuer_token_account: Box<Account<'info, TokenAccount>>,

    // other
    #[account(mut)]
    payer: Signer<'info>,
    token_program: Program<'info, Token>,
    system_program: Program<'info, System>,
}

pub fn handler(ctx: Context<IssueCertificateCtx>, ix: IssueCertificateIx) -> ProgramResult {
    if ix.kind != CertificateKind::Managed as u8 && ix.kind != CertificateKind::Unmanaged as u8{
        return Err(ErrorCode::InvalidCertificateKind.into());
    }
    // increment mint manager
    let mint_manager = &mut ctx.accounts.mint_manager;
    mint_manager.outstanding_certificates += 1;

    // set certificate data
    let certificate = &mut ctx.accounts.certificate;
    certificate.issuer = ctx.accounts.issuer.key();
    certificate.mint = ctx.accounts.certificate_mint.key();
    certificate.amount = ix.amount;
    certificate.recipient = ix.recipient;
    certificate.kind = ix.kind;
    certificate.bump = ix.bump;
    // price
    certificate.payment_amount = ix.payment_amount;
    certificate.payment_mint = ix.payment_mint;
    // lifecycle
    certificate.state = CertificateState::Issued as u8;
    certificate.issued_at = Clock::get().unwrap().unix_timestamp;
    // time expiry
    certificate.duration = ix.duration;
    certificate.start_at_issuance = ix.start_at_issuance;
    // set expiration of issue time + duration 
    if certificate.duration != None && certificate.start_at_issuance != None && certificate.start_at_issuance.unwrap() {
        certificate.expiration = Some(certificate.issued_at + certificate.duration.unwrap());
    }
    // usage expiry
    certificate.max_usages = ix.total_usages;
    certificate.total_usages = ix.total_usages;
    certificate.usages = 0;
    // manual expiry
    certificate.revoke_authority = ix.revoke_authority;
    certificate.is_returnable = ix.is_returnable;
    certificate.is_extendable = ix.is_extendable;

    // transfer token to certificate token account
    let cpi_accounts = Transfer {
        from: ctx.accounts.issuer_token_account.to_account_info(),
        to: ctx.accounts.certificate_token_account.to_account_info(),
        authority: ctx.accounts.issuer.to_account_info(),
    };
    let cpi_program = ctx.accounts.token_program.to_account_info();
    let cpi_context = CpiContext::new(cpi_program, cpi_accounts);
    token::transfer(cpi_context, certificate.amount)?;
    return Ok(())
}