use {
    crate::{state::*, errors::*},
    anchor_lang::{{prelude::*}, AccountsClose}
};
use anchor_spl::{
    token::{self, Token, TokenAccount, Transfer, Mint, ThawAccount, CloseAccount},
};

#[derive(Accounts)]
pub struct InvalidateCertificateCtx<'info> {
    #[account(seeds = [MINT_MANAGER_SEED.as_bytes(), certificate.mint.as_ref()], bump = mint_manager.bump)]
    pub mint_manager: Box<Account<'info, MintManager>>, 
    #[account(mut, constraint =
        certificate.expiration != None && Clock::get().unwrap().unix_timestamp >= certificate.expiration.unwrap()
        || certificate.max_usages != None && certificate.usages >= certificate.max_usages.unwrap()
        @ ErrorCode::CannotInvalidate,
        seeds=[CERTIFICATE_SEED.as_bytes(), certificate.mint.as_ref()], bump=certificate.bump
    )]
    pub certificate: Box<Account<'info, Certificate>>,
    #[account(mut, constraint = certificate_mint.key() == certificate.mint)]
    pub certificate_mint: Box<Account<'info, Mint>>,
    #[account(mut, constraint = certificate_token_account.owner == certificate.key() @ ErrorCode::InvalidCertificateTokenAccount)]
    pub certificate_token_account: Box<Account<'info, TokenAccount>>,
    #[account(mut, constraint = certificate.payment_mint == None || (certificate_payment_token_account.owner == certificate.key() && certificate_payment_token_account.mint == certificate.payment_mint.unwrap()) @ ErrorCode::InvalidCertificatePaymentTokenAccount)]
    pub certificate_payment_token_account: Box<Account<'info, TokenAccount>>,

    // recipient
    #[account(mut)]
    pub recipient_token_account: Box<Account<'info, TokenAccount>>,

    // issuer
    #[account(mut, constraint = issuer_token_account.owner == certificate.issuer @ ErrorCode::InvalidOwnership)]
    pub issuer_token_account: Box<Account<'info, TokenAccount>>,
    #[account(mut, constraint = issuer_payment_token_account.owner == certificate.issuer @ ErrorCode::InvalidIssuerPaymentTokenAccount)]
    pub issuer_payment_token_account: Box<Account<'info, TokenAccount>>,

    // other
    #[account(mut)]
    pub invalidator: Signer<'info>,
    #[account(mut, constraint = invalidator_token_account.owner == invalidator.key() @ ErrorCode::InvalidOwnership)]
    pub invalidator_token_account: Box<Account<'info, TokenAccount>>,
    pub token_program: Program<'info, Token>,
}

pub fn handler(ctx: Context<InvalidateCertificateCtx>) -> ProgramResult {
    let certificate = &mut ctx.accounts.certificate;
    let certificate_state = certificate.state;
    certificate.invalidated_at = Clock::get().unwrap().unix_timestamp;
    certificate.state = CertificateState::Invalidated as u8;

    // get PDA seeds to sign with
    let mint_manager_seeds = &[MINT_MANAGER_SEED.as_bytes(), certificate.mint.as_ref(), &[ctx.accounts.mint_manager.bump]];
    let mint_manager_signer = &[&mint_manager_seeds[..]];
    let certificate_seeds = &[CERTIFICATE_SEED.as_bytes(), certificate.mint.as_ref(), &[certificate.bump]];
    let certificate_signer = &[&certificate_seeds[..]];

    // if claimed
    if certificate_state == CertificateState::Claimed as u8 {
        // transfer price back to issuer
        if certificate.payment_amount != None && certificate.payment_amount.unwrap() > 0 {
            // transfer amount to payment token account
            let cpi_accounts = Transfer {
                from: ctx.accounts.certificate_payment_token_account.to_account_info(),
                to: ctx.accounts.issuer_payment_token_account.to_account_info(),
                authority: certificate.to_account_info(),
            };
            let cpi_program = ctx.accounts.token_program.to_account_info();
            let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
            token::transfer(cpi_context, certificate.payment_amount.unwrap())?;
        }

        // if managed we need to thaw and transfer
        if certificate.kind == CertificateKind::Managed as u8 {
            // thaw recipient account
            let cpi_accounts = ThawAccount {
                account: ctx.accounts.recipient_token_account.to_account_info(),
                mint: ctx.accounts.certificate_mint.to_account_info(),
                authority: ctx.accounts.mint_manager.to_account_info(),
            };
            let cpi_program = ctx.accounts.token_program.to_account_info();
            let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(mint_manager_signer);
            token::thaw_account(cpi_context)?;

            // transfer back to issuer
            let cpi_accounts = Transfer {
                from: ctx.accounts.recipient_token_account.to_account_info(),
                to: ctx.accounts.issuer_token_account.to_account_info(),
                authority: certificate.to_account_info(),
            };
            let cpi_program = ctx.accounts.token_program.to_account_info();
            let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
            token::transfer(cpi_context, certificate.amount)?;

            // close certificate account
            certificate.close(ctx.accounts.invalidator.to_account_info())?;
        }
    } else {
        if certificate.kind == CertificateKind::Managed as u8 {
            // transfer back to issuer
            let cpi_accounts = Transfer {
                from: ctx.accounts.certificate_token_account.to_account_info(),
                to: ctx.accounts.issuer_token_account.to_account_info(),
                authority: certificate.to_account_info(),
            };
            let cpi_program = ctx.accounts.token_program.to_account_info();
            let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
            token::transfer(cpi_context, certificate.amount)?;
            
            // close certificate account
            certificate.close(ctx.accounts.invalidator.to_account_info())?;
        } else {
            // transfer to invalidator
            let cpi_accounts = Transfer {
                from: ctx.accounts.certificate_token_account.to_account_info(),
                to: ctx.accounts.invalidator_token_account.to_account_info(),
                authority: certificate.to_account_info(),
            };
            let cpi_program = ctx.accounts.token_program.to_account_info();
            let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
            token::transfer(cpi_context, certificate.amount)?;
        }
    }

    // close certificate_token_account
    let cpi_accounts = CloseAccount {
        account: ctx.accounts.certificate_token_account.to_account_info(),
        destination: ctx.accounts.invalidator.to_account_info(),
        authority: certificate.to_account_info(),
    };
    let cpi_program = ctx.accounts.token_program.to_account_info();
    let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
    token::close_account(cpi_context)?;

    // close certificate_payment_token_account
    let cpi_accounts = CloseAccount {
        account: ctx.accounts.certificate_payment_token_account.to_account_info(),
        destination: ctx.accounts.invalidator.to_account_info(),
        authority: certificate.to_account_info(),
    };
    let cpi_program = ctx.accounts.token_program.to_account_info();
    let cpi_context = CpiContext::new(cpi_program, cpi_accounts).with_signer(certificate_signer);
    token::close_account(cpi_context)?;

    // decrement mint_manager
    let mint_manager = &mut ctx.accounts.mint_manager;
    mint_manager.outstanding_certificates -= 1;
    return Ok(())
}