use crate::command;
use crate::conf;
use regex::Regex;
use std::process;

pub fn clean() {
    if conf::exist_camrc() {
        conf::delete_camrc().unwrap_or_else(|err| {
            eprintln!("[CAM Error]: {}", err);
            process::exit(1);
        });
        conf::unmount_camrc_env().unwrap_or_else(|err| {
            eprintln!("[CAM Error]: {}", err);
            process::exit(1);
        });
    }
    println!("[CAM Info]: Cam cleanup succeeded")
}

pub fn init() {
    if !conf::exist_camrc() {
        conf::mount_camrc_env().unwrap_or_else(|err| {
            eprintln!("[CAM ERROR]: {}", err);
            process::exit(1);
        });
        conf::create_camrc().unwrap_or_else(|err| {
            eprintln!("[CAM ERROR]: {}", err);
            process::exit(1);
        });
        let env_path = conf::get_env_path().unwrap();
        println!(
            "[CAM INFO]: cam initialization succeeded, please run `source {}`",
            env_path
        )
    } else {
        println!("[CAM INFO]: cam has been initialized")
    }
}

pub fn ls() {
    if conf::exist_camrc() {
        let file_content = conf::read_camrc().unwrap_or_else(|err| {
            eprintln!("[CAM ERROR]: {}", err);
            process::exit(1);
        });
        let reg = Regex::new(r"alias ([0-9a-zA-Z_]*)='([0-9a-zA-Z_ ]*)'").unwrap();
        println!("");
        for caps in reg.captures_iter(file_content.as_str()) {
            println!(
                "     {:-<12} {}",
                caps.get(1).unwrap().as_str().to_owned() + " ",
                caps.get(2).unwrap().as_str(),
            )
        }
        println!("");
    } else {
        println!("[CAM INFO]: cam is not initialized, please run 'cam init'");
    }
}

pub fn add(args: &command::Add) {
    if conf::exist_camrc() {
        if check_name_exist(&args.name) {
            println!("[CAM INFO]: {} alias already exists", args.name);
            process::exit(1);
        }
        let alias_shell = "alias ".to_owned() + &args.name + "='" + &args.shell + "'";
        conf::append_camrc(alias_shell).unwrap_or_else(|err| {
            eprintln!("[CAM ERROR]: {}", err);
            process::exit(1);
        });
        let env_path = conf::get_env_path().unwrap();
        println!(
            "[CAM INFO]: {} added successfully, please run `source {}`",
            args.name, env_path
        );
    } else {
        println!("[CAM INFO]: cam is not initialized, please run 'cam init'");
    }
}

pub fn remove(args: &command::Remove) {
    if conf::exist_camrc() {
        if check_name_exist(&args.name) {
            let file_content = conf::read_camrc().unwrap_or_else(|err| {
                eprintln!("[CAM ERROR]: {}", err);
                process::exit(1);
            });
            let reg = Regex::new(r"alias ([0-9a-zA-Z_]*)='([0-9a-zA-Z_ ]*)'").unwrap();
            let remove_alias = reg
                .captures_iter(file_content.as_str())
                .filter(|cap| cap.get(1).unwrap().as_str() == args.name)
                .map(|cap| cap.get(0).unwrap().as_str())
                .collect::<Vec<_>>();
            for alias in remove_alias {
                conf::remove_camrc(alias).unwrap_or_else(|err| {
                    eprintln!("[CAM ERROR]: {}", err);
                    process::exit(1);
                })
            }
            let env_path = conf::get_env_path().unwrap();
            println!(
                "[CAM INFO]: remove {} alias succeeded, please run `source {}`",
                args.name, env_path
            );
        }
    } else {
        println!("[CAM INFO]: cam is not initialized, please run 'cam init'");
    }
}

fn check_name_exist(name: &String) -> bool {
    let file_content = conf::read_camrc().unwrap_or_else(|err| {
        eprintln!("[CAM ERROR]: {}", err);
        process::exit(1);
    });
    let reg = Regex::new(r"alias ([0-9a-zA-Z_]*)='([0-9a-zA-Z_ ]*)'").unwrap();
    let filter_res = reg
        .captures_iter(file_content.as_str())
        .map(|cap| cap.get(1).unwrap().as_str())
        .filter(|&cap_name| cap_name == name)
        .collect::<Vec<_>>();
    filter_res.len() > 0
}
