use std::collections::HashMap;
use std::sync::RwLock;

use cirru_parser::Cirru;

use crate::data::cirru::code_to_calcit;
use crate::primes::{Calcit, ImportRule};
use crate::snapshot;
use crate::snapshot::Snapshot;
use crate::util::string::extract_pkg_from_def;

pub type ProgramEvaledData = HashMap<Box<str>, HashMap<Box<str>, Calcit>>;

/// information extracted from snapshot
#[derive(Debug, Clone, PartialEq, Eq)]
pub struct ProgramFileData {
  pub import_map: HashMap<Box<str>, ImportRule>,
  pub defs: HashMap<Box<str>, Calcit>,
}

pub type ProgramCodeData = HashMap<Box<str>, ProgramFileData>;

lazy_static! {
  /// data of program running
  static ref PROGRAM_EVALED_DATA_STATE: RwLock<ProgramEvaledData> = RwLock::new(HashMap::new());
  /// raw code information before program running
  pub static ref PROGRAM_CODE_DATA: RwLock<ProgramCodeData> = RwLock::new(HashMap::new());
}

fn extract_import_rule(nodes: &Cirru) -> Result<Vec<(Box<str>, ImportRule)>, String> {
  match nodes {
    Cirru::Leaf(_) => Err(String::from("Expected import rule in expr")),
    Cirru::List(rule_nodes) => {
      let mut xs = rule_nodes.to_owned();
      match xs.get(0) {
        // strip leading `[]` symbols
        Some(Cirru::Leaf(s)) if &**s == "[]" => xs = xs[1..4].to_vec(),
        _ => (),
      }
      match (xs[0].to_owned(), xs[1].to_owned(), xs[2].to_owned()) {
        (Cirru::Leaf(ns), x, Cirru::Leaf(alias)) if x == Cirru::leaf(":as") => Ok(vec![(alias, ImportRule::NsAs(ns))]),
        (Cirru::Leaf(ns), x, Cirru::Leaf(alias)) if x == Cirru::leaf(":default") => Ok(vec![(alias, ImportRule::NsDefault(ns))]),
        (Cirru::Leaf(ns), x, Cirru::List(ys)) if x == Cirru::leaf(":refer") => {
          let mut rules: Vec<(Box<str>, ImportRule)> = Vec::with_capacity(ys.len());
          for y in ys {
            match y {
              Cirru::Leaf(s) if &*s == "[]" => (), // `[]` symbol are ignored
              Cirru::Leaf(s) => rules.push((s.to_owned(), ImportRule::NsReferDef(ns.to_owned(), s.to_owned()))),
              Cirru::List(_defs) => return Err(String::from("invalid refer values")),
            }
          }
          Ok(rules)
        }
        (_, x, _) if x == Cirru::leaf(":as") => Err(String::from("invalid import rule")),
        (_, x, _) if x == Cirru::leaf(":default") => Err(String::from("invalid default rule")),
        (_, x, _) if x == Cirru::leaf(":refer") => Err(String::from("invalid import rule")),
        _ if xs.len() != 3 => Err(format!("expected import rule has length 3: {}", Cirru::List(xs.to_owned()))),
        _ => Err(String::from("unknown rule")),
      }
    }
  }
}

fn extract_import_map(nodes: &Cirru) -> Result<HashMap<Box<str>, ImportRule>, String> {
  match nodes {
    Cirru::Leaf(_) => unreachable!("Expected expr for ns"),
    Cirru::List(xs) => match (xs.get(0), xs.get(1), xs.get(2)) {
      // Too many clones
      (Some(x), Some(Cirru::Leaf(_)), Some(Cirru::List(xs))) if *x == Cirru::leaf("ns") => {
        if !xs.is_empty() && xs[0] == Cirru::leaf(":require") {
          let mut ys: HashMap<Box<str>, ImportRule> = HashMap::with_capacity(xs.len());
          for (idx, x) in xs.iter().enumerate() {
            if idx > 0 {
              let rules = extract_import_rule(x)?;
              for (target, rule) in rules {
                ys.insert(target, rule);
              }
            }
          }
          Ok(ys)
        } else {
          Ok(HashMap::new())
        }
      }
      _ if xs.len() < 3 => Ok(HashMap::new()),
      _ => Err(String::from("invalid ns form")),
    },
  }
}

fn extract_file_data(file: snapshot::FileInSnapShot, ns: Box<str>) -> Result<ProgramFileData, String> {
  let import_map = extract_import_map(&file.ns)?;
  let mut defs: HashMap<Box<str>, Calcit> = HashMap::with_capacity(file.defs.len());
  for (def, code) in file.defs {
    let at_def = def.to_owned();
    defs.insert(def, code_to_calcit(&code, &ns, &at_def)?);
  }
  Ok(ProgramFileData { import_map, defs })
}

pub fn extract_program_data(s: &Snapshot) -> Result<ProgramCodeData, String> {
  let mut xs: ProgramCodeData = HashMap::with_capacity(s.files.len());
  for (ns, file) in s.files.to_owned() {
    let file_info = extract_file_data(file, ns.to_owned())?;
    xs.insert(ns, file_info);
  }
  Ok(xs)
}

// lookup without cloning
pub fn has_def_code(ns: &str, def: &str) -> bool {
  let program_code = { PROGRAM_CODE_DATA.read().unwrap() };
  match program_code.get(ns) {
    Some(v) => v.defs.contains_key(def),
    None => false,
  }
}

pub fn lookup_def_code(ns: &str, def: &str) -> Option<Calcit> {
  let program_code = { PROGRAM_CODE_DATA.read().unwrap() };
  let file = program_code.get(ns)?;
  let data = file.defs.get(def)?;
  Some(data.to_owned())
}

pub fn lookup_def_target_in_import(ns: &str, def: &str) -> Option<Box<str>> {
  let program = { PROGRAM_CODE_DATA.read().unwrap() };
  let file = program.get(ns)?;
  let import_rule = file.import_map.get(def)?;
  match import_rule {
    ImportRule::NsReferDef(ns, _def) => Some(ns.to_owned()),
    ImportRule::NsAs(_ns) => None,
    ImportRule::NsDefault(_ns) => None,
  }
}

pub fn lookup_ns_target_in_import(ns: &str, alias: &str) -> Option<Box<str>> {
  let program = { PROGRAM_CODE_DATA.read().unwrap() };
  let file = program.get(ns)?;
  let import_rule = file.import_map.get(alias)?;
  match import_rule {
    ImportRule::NsReferDef(_ns, _def) => None,
    ImportRule::NsAs(ns) => Some(ns.to_owned()),
    ImportRule::NsDefault(_ns) => None,
  }
}

// imported via :default
pub fn lookup_default_target_in_import(ns: &str, alias: &str) -> Option<Box<str>> {
  let program = { PROGRAM_CODE_DATA.read().unwrap() };
  let file = program.get(ns)?;
  let import_rule = file.import_map.get(alias)?;
  match import_rule {
    ImportRule::NsReferDef(_ns, _def) => None,
    ImportRule::NsAs(_ns) => None,
    ImportRule::NsDefault(ns) => Some(ns.to_owned()),
  }
}

/// similar to lookup, but skipped cloning
#[allow(dead_code)]
pub fn has_evaled_def(ns: &str, def: &str) -> bool {
  let s2 = PROGRAM_EVALED_DATA_STATE.read().unwrap();
  s2.contains_key(ns) && s2[ns].contains_key(def)
}

/// lookup and return value
pub fn lookup_evaled_def(ns: &str, def: &str) -> Option<Calcit> {
  let s2 = PROGRAM_EVALED_DATA_STATE.read().unwrap();
  if s2.contains_key(ns) && s2[ns].contains_key(def) {
    Some(s2[ns][def].to_owned())
  } else {
    // println!("failed to lookup {} {}", ns, def);
    None
  }
}

// Dirty mutating global states
pub fn write_evaled_def(ns: &str, def: &str, value: Calcit) -> Result<(), String> {
  // println!("writing {} {}", ns, def);
  let mut program = PROGRAM_EVALED_DATA_STATE.write().unwrap();
  if !program.contains_key(ns) {
    (*program).insert(String::from(ns).into_boxed_str(), HashMap::new());
  }

  let file = program.get_mut(ns).unwrap();
  file.insert(String::from(def).into_boxed_str(), value);

  Ok(())
}

// take a snapshot for codegen
pub fn clone_evaled_program() -> ProgramEvaledData {
  let program = &PROGRAM_EVALED_DATA_STATE.read().unwrap();

  let mut xs: ProgramEvaledData = HashMap::new();
  for k in program.keys() {
    xs.insert(k.to_owned(), program[k].to_owned());
  }
  xs
}

pub fn apply_code_changes(changes: &snapshot::ChangesDict) -> Result<(), String> {
  let mut program_code = { PROGRAM_CODE_DATA.write().unwrap() };

  for (ns, file) in &changes.added {
    program_code.insert(ns.to_owned(), extract_file_data(file.to_owned(), ns.to_owned())?);
  }
  for ns in &changes.removed {
    program_code.remove(ns);
  }
  for (ns, info) in &changes.changed {
    // println!("handling ns: {:?} {}", ns, program_code.contains_key(ns));
    let file = program_code.get_mut(ns).unwrap();
    if info.ns.is_some() {
      file.import_map = extract_import_map(&info.ns.to_owned().unwrap())?;
    }
    for (def, code) in &info.added_defs {
      file.defs.insert(def.to_owned(), code_to_calcit(code, ns, def)?);
    }
    for def in &info.removed_defs {
      file.defs.remove(def);
    }
    for (def, code) in &info.changed_defs {
      file.defs.insert(def.to_owned(), code_to_calcit(code, ns, def)?);
    }
  }

  Ok(())
}

/// clear evaled data after reloading
pub fn clear_all_program_evaled_defs(init_fn: &str, reload_fn: &str, reload_libs: bool) -> Result<(), String> {
  let mut program = PROGRAM_EVALED_DATA_STATE.write().unwrap();
  if reload_libs {
    (*program).clear();
  } else {
    // reduce changes of libs. could be dirty in some cases
    let init_pkg = extract_pkg_from_def(init_fn).unwrap();
    let reload_pkg = extract_pkg_from_def(reload_fn).unwrap();
    let mut to_remove: Vec<Box<str>> = vec![];
    for k in (*program).keys() {
      if k == &init_pkg || k == &reload_pkg || k.starts_with(&format!("{}.", init_pkg)) || k.starts_with(&format!("{}.", reload_pkg)) {
        to_remove.push(k.to_owned());
      } else {
        continue;
      }
    }
    for k in to_remove {
      (*program).remove(&k);
    }
  }
  Ok(())
}
