// CALCIT VERSION
export const calcit_version = "0.3.35";

import { overwriteComparator, initTernaryTreeMap } from "@calcit/ternary-tree";
import { parse } from "@cirru/parser.ts";

import { CrDataValue } from "./js-primes";
import {
  CrDataSymbol,
  CrDataKeyword,
  CrDataRef,
  CrDataFn,
  CrDataRecur,
  kwd,
  refsRegistry,
  toString,
  getStringName,
  to_js_data,
  _AND__EQ_,
} from "./calcit-data";

import { fieldsEqual, CrDataRecord } from "./js-record";

export * from "./calcit-data";
export * from "./js-record";
export * from "./js-map";
export * from "./js-list";
export * from "./js-set";
export * from "./js-primes";
export * from "./js-tuple";
export * from "./custom-formatter";
export * from "./js-cirru";

import { CrDataList, foldl } from "./js-list";
import { CrDataMap } from "./js-map";
import { CrDataSet } from "./js-set";
import { CrDataTuple } from "./js-tuple";
import { to_calcit_data, extract_cirru_edn } from "./js-cirru";

let inNodeJs = typeof process !== "undefined" && process?.release?.name === "node";

export let type_of = (x: any): CrDataKeyword => {
  if (typeof x === "string") {
    return kwd("string");
  }
  if (typeof x === "number") {
    return kwd("number");
  }
  if (x instanceof CrDataKeyword) {
    return kwd("keyword");
  }
  if (x instanceof CrDataList) {
    return kwd("list");
  }
  if (x instanceof CrDataMap) {
    return kwd("map");
  }
  if (x == null) {
    return kwd("nil");
  }
  if (x instanceof CrDataRef) {
    return kwd("ref");
  }
  if (x instanceof CrDataTuple) {
    return kwd("tuple");
  }
  if (x instanceof CrDataSymbol) {
    return kwd("symbol");
  }
  if (x instanceof CrDataSet) {
    return kwd("set");
  }
  if (x instanceof CrDataRecord) {
    return kwd("record");
  }
  if (x === true || x === false) {
    return kwd("bool");
  }
  if (typeof x === "function") {
    if (x.isMacro) {
      // this is faked...
      return kwd("macro");
    }
    return kwd("fn");
  }
  if (typeof x === "object") {
    return kwd("js-object");
  }
  throw new Error(`Unknown data ${x}`);
};

export let print = (...xs: CrDataValue[]): void => {
  // TODO stringify each values
  console.log(xs.map((x) => toString(x, false)).join(" "));
};

export function _AND_list_COL_count(x: CrDataValue): number {
  if (x instanceof CrDataList) return x.len();

  throw new Error(`expected a list ${x}`);
}
export function _AND_str_COL_count(x: CrDataValue): number {
  if (typeof x === "string") return x.length;

  throw new Error(`expected a string ${x}`);
}
export function _AND_map_COL_count(x: CrDataValue): number {
  if (x instanceof CrDataMap) return x.len();

  throw new Error(`expected a map ${x}`);
}
export function _AND_record_COL_count(x: CrDataValue): number {
  if (x instanceof CrDataRecord) return x.fields.length;

  throw new Error(`expected a record ${x}`);
}
export function _AND_set_COL_count(x: CrDataValue): number {
  if (x instanceof CrDataSet) return x.len();

  throw new Error(`expected a set ${x}`);
}

export let _LIST_ = (...xs: CrDataValue[]): CrDataList => {
  return new CrDataList(xs);
};
// single quote as alias for list
export let _SQUO_ = (...xs: CrDataValue[]): CrDataList => {
  return new CrDataList(xs);
};

export let _AND__MAP_ = (...xs: CrDataValue[]): CrDataMap => {
  if (xs.length % 2 !== 0) {
    throw new Error("&map expects even number of arguments");
  }
  return new CrDataMap(xs);
};

export let _AND_list_map = (...xs: CrDataValue[]): CrDataList => {
  if (xs.length != 2) {
    throw new Error("&list-map expected 2 arguments");
  }
  if (typeof xs[0] !== "function") {
    throw new Error("&list-map expected a function");
  }
  let f = xs[0];
  if (!(xs[1] instanceof CrDataList)) {
    throw new Error("&list-map expected a list");
  }
  //Array.prototype.map proviced 3 arguments, only one is needed
  return xs[1].map((x) => f(x));
};

export let defatom = (path: string, x: CrDataValue): CrDataValue => {
  let v = new CrDataRef(x, path);
  refsRegistry.set(path, v);
  return v;
};

export let peekDefatom = (path: string): CrDataRef => {
  return refsRegistry.get(path);
};

export let deref = (x: CrDataRef): CrDataValue => {
  let a = refsRegistry.get(x.path);
  if (!(a instanceof CrDataRef)) {
    console.warn("Can not find ref:", x);
  }
  return a.value;
};

export let _AND__ADD_ = (x: number, y: number): number => {
  return x + y;
};

export let _AND__STAR_ = (x: number, y: number): number => {
  return x * y;
};

export let _AND_str = (x: CrDataValue): string => {
  return `${x}`;
};

export let _AND_str_COL_contains_QUES_ = (xs: CrDataValue, x: CrDataValue): boolean => {
  if (typeof xs === "string") {
    if (typeof x != "number") {
      throw new Error("Expected number index for detecting");
    }
    let size = xs.length;
    if (x >= 0 && x < size) {
      return true;
    }
    return false;
  }

  throw new Error("string `contains?` expected a string");
};

export let _AND_list_COL_contains_QUES_ = (xs: CrDataValue, x: CrDataValue): boolean => {
  if (xs instanceof CrDataList) {
    if (typeof x != "number") {
      throw new Error("Expected number index for detecting");
    }
    let size = xs.len();
    if (x >= 0 && x < size) {
      return true;
    }
    return false;
  }

  throw new Error("list `contains?` expected a list");
};

export let _AND_map_COL_contains_QUES_ = (xs: CrDataValue, x: CrDataValue): boolean => {
  if (xs instanceof CrDataMap) return xs.contains(x);

  throw new Error("map `contains?` expected a map");
};

export let _AND_record_COL_contains_QUES_ = (xs: CrDataValue, x: CrDataValue): boolean => {
  if (xs instanceof CrDataRecord) return xs.contains(x);

  throw new Error("record `contains?` expected a record");
};

export let _AND_str_COL_includes_QUES_ = (xs: CrDataValue, x: CrDataValue): boolean => {
  if (typeof xs === "string") {
    if (typeof x !== "string") {
      throw new Error("Expected string");
    }
    return xs.includes(x as string);
  }

  throw new Error("string includes? expected a string");
};

export let _AND_list_COL_includes_QUES_ = (xs: CrDataValue, x: CrDataValue): boolean => {
  if (xs instanceof CrDataList) {
    let size = xs.len();
    for (let v of xs.items()) {
      if (_AND__EQ_(v, x)) {
        return true;
      }
    }
    return false;
  }

  throw new Error("list includes? expected a list");
};

export let _AND_map_COL_includes_QUES_ = (xs: CrDataValue, x: CrDataValue): boolean => {
  if (xs instanceof CrDataMap) {
    for (let [k, v] of xs.pairs()) {
      if (_AND__EQ_(v, x)) {
        return true;
      }
    }
    return false;
  }

  throw new Error("map includes? expected a map");
};

export let _AND_set_COL_includes_QUES_ = (xs: CrDataValue, x: CrDataValue): boolean => {
  if (xs instanceof CrDataSet) {
    return xs.contains(x);
  }

  throw new Error("set includes? expected a set");
};

export let _AND_str_COL_nth = function (xs: CrDataValue, k: CrDataValue) {
  if (arguments.length !== 2) throw new Error("nth takes 2 arguments");
  if (typeof k !== "number") throw new Error("Expected number index for a list");

  if (typeof xs === "string") return xs[k];

  throw new Error("Does not support `nth` on this type");
};

export let _AND_list_COL_nth = function (xs: CrDataValue, k: CrDataValue) {
  if (arguments.length !== 2) throw new Error("nth takes 2 arguments");
  if (typeof k !== "number") throw new Error("Expected number index for a list");

  if (xs instanceof CrDataList) return xs.get(k);

  throw new Error("Does not support `nth` on this type");
};

export let _AND_tuple_COL_nth = function (xs: CrDataValue, k: CrDataValue) {
  if (arguments.length !== 2) throw new Error("nth takes 2 arguments");
  if (typeof k !== "number") throw new Error("Expected number index for a list");

  if (xs instanceof CrDataTuple) return xs.get(k);

  throw new Error("Does not support `nth` on this type");
};

export let _AND_record_COL_nth = function (xs: CrDataValue, k: CrDataValue) {
  if (arguments.length !== 2) throw new Error("nth takes 2 arguments");
  if (typeof k !== "number") throw new Error("Expected number index for a list");

  if (xs instanceof CrDataRecord) {
    if (k < 0 || k >= xs.fields.length) {
      throw new Error("Out of bound");
    }
    return new CrDataList([kwd(xs.fields[k]), xs.values[k]]);
  }

  throw new Error("Does not support `nth` on this type");
};

export let _AND_record_COL_get = function (xs: CrDataValue, k: CrDataValue) {
  if (arguments.length !== 2) {
    throw new Error("record &get takes 2 arguments");
  }

  if (xs instanceof CrDataRecord) return xs.get(k);

  throw new Error("Does not support `&get` on this type");
};

export let _AND_list_COL_assoc = function (xs: CrDataValue, k: CrDataValue, v: CrDataValue) {
  if (arguments.length !== 3) throw new Error("assoc takes 3 arguments");

  if (xs instanceof CrDataList) {
    if (typeof k !== "number") {
      throw new Error("Expected number index for lists");
    }
    return xs.assoc(k, v);
  }
  throw new Error("list `assoc` expected a list");
};
export let _AND_tuple_COL_assoc = function (xs: CrDataValue, k: CrDataValue, v: CrDataValue) {
  if (arguments.length !== 3) throw new Error("assoc takes 3 arguments");

  if (xs instanceof CrDataTuple) {
    if (typeof k !== "number") {
      throw new Error("Expected number index for lists");
    }
    return xs.assoc(k, v);
  }

  throw new Error("tuple `assoc` expected a tuple");
};
export let _AND_map_COL_assoc = function (xs: CrDataValue, k: CrDataValue, v: CrDataValue) {
  if (arguments.length !== 3) throw new Error("assoc takes 3 arguments");

  if (xs instanceof CrDataMap) return xs.assoc(k, v);

  throw new Error("map `assoc` expected a map");
};
export let _AND_record_COL_assoc = function (xs: CrDataValue, k: CrDataValue, v: CrDataValue) {
  if (arguments.length !== 3) throw new Error("assoc takes 3 arguments");

  if (xs instanceof CrDataRecord) return xs.assoc(k, v);

  throw new Error("record `assoc` expected a record");
};

export let assoc_before = function (xs: CrDataList, k: number, v: CrDataValue): CrDataList {
  if (arguments.length !== 3) {
    throw new Error("assoc takes 3 arguments");
  }
  if (xs instanceof CrDataList) {
    if (typeof k !== "number") {
      throw new Error("Expected number index for lists");
    }
    return xs.assocBefore(k, v);
  }

  throw new Error("Does not support `assoc-before` on this type");
};

export let assoc_after = function (xs: CrDataList, k: number, v: CrDataValue): CrDataList {
  if (arguments.length !== 3) {
    throw new Error("assoc takes 3 arguments");
  }
  if (xs instanceof CrDataList) {
    if (typeof k !== "number") {
      throw new Error("Expected number index for lists");
    }
    return xs.assocAfter(k, v);
  }

  throw new Error("Does not support `assoc-after` on this type");
};

export let _AND_list_COL_dissoc = function (xs: CrDataValue, k: CrDataValue) {
  if (arguments.length !== 2) throw new Error("dissoc takes 2 arguments");

  if (xs instanceof CrDataList) {
    if (typeof k !== "number") throw new Error("Expected number index for lists");

    return xs.dissoc(k);
  }

  throw new Error("`dissoc` expected a list");
};
export let _AND_map_COL_dissoc = function (xs: CrDataValue, k: CrDataValue) {
  if (arguments.length !== 2) throw new Error("dissoc takes 2 arguments");

  if (xs instanceof CrDataMap) return xs.dissoc(k);

  throw new Error("`dissoc` expected a map");
};

export let reset_BANG_ = (a: CrDataRef, v: CrDataValue): null => {
  if (!(a instanceof CrDataRef)) {
    throw new Error("Expected ref for reset!");
  }
  let prev = a.value;
  a.value = v;
  for (let [k, f] of a.listeners) {
    f(v, prev);
  }
  return null;
};

export let add_watch = (a: CrDataRef, k: CrDataKeyword, f: CrDataFn): null => {
  if (!(a instanceof CrDataRef)) {
    throw new Error("Expected ref for add-watch!");
  }
  if (!(k instanceof CrDataKeyword)) {
    throw new Error("Expected watcher key in keyword");
  }
  if (!(typeof f === "function")) {
    throw new Error("Expected watcher function");
  }
  a.listeners.set(k, f);
  return null;
};

export let remove_watch = (a: CrDataRef, k: CrDataKeyword): null => {
  a.listeners.delete(k);
  return null;
};

export let range = (n: number, m: number, m2: number): CrDataList => {
  var result = new CrDataList([]);
  if (m2 != null) {
    console.warn("TODO range with 3 arguments"); // TODO
  }
  if (m != null) {
    var idx = n;
    while (idx < m) {
      result = result.append(idx);
      idx = idx + 1;
    }
  } else {
    var idx = 0;
    while (idx < n) {
      result = result.append(idx);
      idx = idx + 1;
    }
  }
  return result;
};

export function _AND_list_COL_empty_QUES_(xs: CrDataValue): boolean {
  if (xs instanceof CrDataList) return xs.isEmpty();
  throw new Error(`expected a list ${xs}`);
}
export function _AND_str_COL_empty_QUES_(xs: CrDataValue): boolean {
  if (typeof xs == "string") return xs.length == 0;
  throw new Error(`expected a string ${xs}`);
}
export function _AND_map_COL_empty_QUES_(xs: CrDataValue): boolean {
  if (xs instanceof CrDataMap) return xs.isEmpty();

  throw new Error(`expected a list ${xs}`);
}
export function _AND_set_COL_empty_QUES_(xs: CrDataValue): boolean {
  if (xs instanceof CrDataSet) return xs.len() === 0;
  throw new Error(`expected a list ${xs}`);
}

export let _AND_list_COL_first = (xs: CrDataValue): CrDataValue => {
  if (xs instanceof CrDataList) {
    if (xs.isEmpty()) {
      return null;
    }
    return xs.first();
  }
  console.error(xs);
  throw new Error("Expected a list");
};
export let _AND_str_COL_first = (xs: CrDataValue): CrDataValue => {
  if (typeof xs === "string") {
    return xs[0];
  }
  console.error(xs);
  throw new Error("Expected a string");
};
export let _AND_map_COL_first = (xs: CrDataValue): CrDataValue => {
  if (xs instanceof CrDataMap) {
    // TODO order may not be stable enough
    let ys = xs.pairs();
    if (ys.length > 0) {
      return new CrDataList(ys[0]);
    } else {
      return null;
    }
  }
  console.error(xs);
  throw new Error("Expected a map");
};
export let _AND_set_COL_first = (xs: CrDataValue): CrDataValue => {
  if (xs instanceof CrDataSet) {
    return xs.first();
  }

  console.error(xs);
  throw new Error("Expected a set");
};

export let timeout_call = (duration: number, f: CrDataFn): null => {
  if (typeof duration !== "number") {
    throw new Error("Expected duration in number");
  }
  if (typeof f !== "function") {
    throw new Error("Expected callback in fn");
  }
  setTimeout(f, duration);
  return null;
};

export let _AND_list_COL_rest = (xs: CrDataValue): CrDataValue => {
  if (xs instanceof CrDataList) {
    if (xs.len() === 0) {
      return null;
    }
    return xs.rest();
  }
  console.error(xs);
  throw new Error("Expected a list");
};

export let _AND_str_COL_rest = (xs: CrDataValue): CrDataValue => {
  if (typeof xs === "string") return xs.substr(1);

  console.error(xs);
  throw new Error("Expects a string");
};
export let _AND_set_COL_rest = (xs: CrDataValue): CrDataValue => {
  if (xs instanceof CrDataSet) return xs.rest();

  console.error(xs);
  throw new Error("Expect a set");
};
export let _AND_map_COL_rest = (xs: CrDataValue): CrDataValue => {
  if (xs instanceof CrDataMap) {
    if (xs.len() > 0) {
      let k0 = xs.pairs()[0][0];
      return xs.dissoc(k0);
    } else {
      return new CrDataMap(initTernaryTreeMap<CrDataValue, CrDataValue>([]));
    }
  }
  console.error(xs);
  throw new Error("Expected map");
};

export let recur = (...xs: CrDataValue[]): CrDataRecur => {
  return new CrDataRecur(xs);
};

export let _AND_get_calcit_backend = () => {
  return kwd("js");
};

export let not = (x: boolean): boolean => {
  return !x;
};

export let prepend = (xs: CrDataValue, v: CrDataValue): CrDataList => {
  if (!(xs instanceof CrDataList)) {
    throw new Error("Expected array");
  }
  return xs.prepend(v);
};

export let append = (xs: CrDataValue, v: CrDataValue): CrDataList => {
  if (!(xs instanceof CrDataList)) {
    throw new Error("Expected array");
  }
  return xs.append(v);
};

export let last = (xs: CrDataValue): CrDataValue => {
  if (xs instanceof CrDataList) {
    if (xs.isEmpty()) {
      return null;
    }
    return xs.get(xs.len() - 1);
  }
  if (typeof xs === "string") {
    return xs[xs.length - 1];
  }
  console.error(xs);
  throw new Error("Data not ready for last");
};

export let butlast = (xs: CrDataValue): CrDataValue => {
  if (xs instanceof CrDataList) {
    if (xs.len() === 0) {
      return null;
    }
    return xs.slice(0, xs.len() - 1);
  }
  if (typeof xs === "string") {
    return xs.substr(0, xs.length - 1);
  }
  console.error(xs);
  throw new Error("Data not ready for butlast");
};

export let initCrTernary = (x: string): CrDataValue => {
  console.error("Ternary for js not implemented yet!");
  return null;
};

export let _SHA__MAP_ = (...xs: CrDataValue[]): CrDataValue => {
  var result = new Set<CrDataValue>();
  for (let idx in xs) {
    result.add(xs[idx]);
  }
  return new CrDataSet(result);
};

let idCounter = 0;

export let generate_id_BANG_ = (): string => {
  // TODO use nanoid.. this code is wrong
  idCounter = idCounter + 1;
  return `gen_id_${idCounter}`;
};

export let display_stack = (): null => {
  console.trace();
  return null;
};

export let slice = (xs: CrDataList, from: number, to: number): CrDataList => {
  if (xs == null) {
    return null;
  }
  let size = xs.len();
  if (to == null) {
    to = size;
  } else if (to <= from) {
    return new CrDataList([]);
  } else if (to > size) {
    to = size;
  }
  return xs.slice(from, to);
};

export let concat = (...lists: CrDataList[]): CrDataList => {
  let result: CrDataList = new CrDataList([]);
  for (let item of lists) {
    if (item == null) {
      continue;
    }
    if (item instanceof CrDataList) {
      if (result.isEmpty()) {
        result = item;
      } else {
        result = result.concat(item);
      }
    } else {
      throw new Error("Expected list for concatenation");
    }
  }
  return result;
};

export let reverse = (xs: CrDataList): CrDataList => {
  if (xs == null) {
    return null;
  }
  return xs.reverse();
};

export let format_ternary_tree = (): null => {
  console.warn("No such function for js");
  return null;
};

export let _AND__GT_ = (a: number, b: number): boolean => {
  return a > b;
};
export let _AND__LT_ = (a: number, b: number): boolean => {
  return a < b;
};
export let _AND__ = (a: number, b: number): number => {
  return a - b;
};
export let _AND__SLSH_ = (a: number, b: number): number => {
  return a / b;
};
export let rem = (a: number, b: number): number => {
  return a % b;
};
export let integer_QUES_ = (a: number) => {
  return a == Math.round(a);
};
export let _AND_str_concat = (a: string, b: string) => {
  return `${toString(a, false)}${toString(b, false)}`;
};
export let sort = (xs: CrDataList, f: CrDataFn): CrDataList => {
  if (xs == null) {
    return null;
  }
  if (xs instanceof CrDataList) {
    let ys = xs.toArray();
    return new CrDataList(ys.sort(f as any));
  }
  throw new Error("Expected list");
};

export let rand = (n: number, m: number): number => {
  if (m != null) {
    return n + (m - n) * Math.random();
  }
  if (n != null) {
    return Math.random() * n;
  }
  return Math.random() * 100;
};

export let rand_int = (n: number, m: number): number => {
  if (m != null) {
    return Math.floor(n + Math.random() * (m - n));
  }
  if (n != null) {
    return Math.floor(Math.random() * n);
  }
  return Math.floor(Math.random() * 100);
};

export let floor = (n: number): number => {
  return Math.floor(n);
};

export let _AND_merge = (a: CrDataValue, b: CrDataMap): CrDataValue => {
  if (a == null) {
    return b;
  }
  if (b == null) {
    return a;
  }
  if (a instanceof CrDataMap) {
    if (b instanceof CrDataMap) {
      return a.merge(b);
    } else {
      throw new Error("Expected an argument of map");
    }
  }
  if (a instanceof CrDataRecord) {
    if (b instanceof CrDataMap) {
      let values = [];
      for (let item of a.values) {
        values.push(item);
      }
      for (let [k, v] of b.pairs()) {
        let field = getStringName(k);
        let idx = a.fields.indexOf(field);
        if (idx >= 0) {
          values[idx] = v;
        } else {
          throw new Error(`Cannot find field ${field} among (${a.fields.join(", ")})`);
        }
      }
      return new CrDataRecord(a.name, a.fields, values);
    }
  }
  throw new Error("Expected map or record");
};

export let _AND_merge_non_nil = (a: CrDataMap, b: CrDataMap): CrDataMap => {
  if (a == null) {
    return b;
  }
  if (b == null) {
    return a;
  }
  if (!(a instanceof CrDataMap)) {
    throw new Error("Expected map");
  }
  if (!(b instanceof CrDataMap)) {
    throw new Error("Expected map");
  }

  return a.mergeSkip(b, null);
};

export let to_pairs = (xs: CrDataValue): CrDataValue => {
  if (xs instanceof CrDataMap) {
    let result: Set<CrDataList> = new Set();
    for (let [k, v] of xs.pairs()) {
      result.add(new CrDataList([k, v]));
    }
    return new CrDataSet(result);
  } else if (xs instanceof CrDataRecord) {
    let arr_result: Array<CrDataList> = [];
    for (let idx in xs.fields) {
      arr_result.push(new CrDataList([kwd(xs.fields[idx]), xs.values[idx]]));
    }
    return new CrDataList(arr_result);
  } else {
    throw new Error("Expected a map");
  }
};

// Math functions

export let sin = (n: number) => {
  return Math.sin(n);
};
export let cos = (n: number) => {
  return Math.cos(n);
};
export let pow = (n: number, m: number) => {
  return Math.pow(n, m);
};
export let ceil = (n: number) => {
  return Math.ceil(n);
};
export let round = (n: number) => {
  return Math.round(n);
};
export let fractional = (n: number) => {
  return n - Math.floor(n);
};
export let sqrt = (n: number) => {
  return Math.sqrt(n);
};

// Set functions

export let _AND_include = (xs: CrDataSet, y: CrDataValue): CrDataSet => {
  if (!(xs instanceof CrDataSet)) {
    throw new Error("Expected a set");
  }
  if (y == null) {
    return xs;
  }
  return xs.include(y);
};

export let _AND_exclude = (xs: CrDataSet, y: CrDataValue): CrDataSet => {
  if (!(xs instanceof CrDataSet)) {
    throw new Error("Expected a set");
  }
  if (y == null) {
    return xs;
  }
  return xs.exclude(y);
};

export let _AND_difference = (xs: CrDataSet, ys: CrDataSet): CrDataSet => {
  if (!(xs instanceof CrDataSet)) {
    throw new Error("Expected a set");
  }
  if (!(ys instanceof CrDataSet)) {
    throw new Error("Expected a set for ys");
  }
  return xs.difference(ys);
};

export let _AND_union = (xs: CrDataSet, ys: CrDataSet): CrDataSet => {
  if (!(xs instanceof CrDataSet)) {
    throw new Error("Expected a set");
  }
  if (!(ys instanceof CrDataSet)) {
    throw new Error("Expected a set for ys");
  }
  return xs.union(ys);
};

export let _AND_intersection = (xs: CrDataSet, ys: CrDataSet): CrDataSet => {
  if (!(xs instanceof CrDataSet)) {
    throw new Error("Expected a set");
  }
  if (!(ys instanceof CrDataSet)) {
    throw new Error("Expected a set for ys");
  }
  return xs.intersection(ys);
};

export let replace = (x: string, y: string, z: string): string => {
  var result = x;
  while (result.indexOf(y) >= 0) {
    result = result.replace(y, z);
  }
  return result;
};

export let split = (xs: string, x: string): CrDataList => {
  return new CrDataList(xs.split(x));
};
export let split_lines = (xs: string): CrDataList => {
  return new CrDataList(xs.split("\n"));
};
export let substr = (xs: string, m: number, n: number): string => {
  if (n <= m) {
    console.warn("endIndex too small");
    return "";
  }
  return xs.substring(m, n);
};

export let str_find = (x: string, y: string): number => {
  return x.indexOf(y);
};

export let parse_float = (x: string): number => {
  return parseFloat(x);
};
export let trim = (x: string, c: string): string => {
  if (c != null) {
    if (c.length !== 1) {
      throw new Error("Expceted c of a character");
    }
    var buffer = x;
    var size = buffer.length;
    var idx = 0;
    while (idx < size && buffer[idx] == c) {
      idx = idx + 1;
    }
    buffer = buffer.substring(idx);
    var size = buffer.length;
    var idx = size;
    while (idx > 1 && buffer[idx - 1] == c) {
      idx = idx - 1;
    }
    buffer = buffer.substring(0, idx);
    return buffer;
  }
  return x.trim();
};

export let format_number = (x: number, n: number): string => {
  return x.toFixed(n);
};

export let get_char_code = (c: string): number => {
  if (typeof c !== "string" || c.length !== 1) {
    throw new Error("Expected a character");
  }
  return c.charCodeAt(0);
};

export let re_matches = (content: string, re: string): boolean => {
  return new RegExp(re).test(content);
};

export let re_find_index = (content: string, re: string): number => {
  return content.search(new RegExp(re));
};

export let re_find_all = (content: string, re: string): CrDataList => {
  let ys = content.match(new RegExp(re, "g"));
  if (ys == null) {
    return new CrDataList([]);
  } else {
    return new CrDataList(ys);
  }
};

export let parse_json = (x: string): CrDataValue => {
  return to_calcit_data(JSON.parse(x), false);
};

export let stringify_json = (x: CrDataValue, addColon: boolean = false): string => {
  return JSON.stringify(to_js_data(x, addColon));
};

export let set__GT_list = (x: CrDataSet): CrDataList => {
  var result: CrDataValue[] = [];
  x.value.forEach((item) => {
    result.push(item);
  });
  return new CrDataList(result);
};

export let aget = (x: any, name: string): any => {
  return x[name];
};
export let aset = (x: any, name: string, v: any): any => {
  return (x[name] = v);
};

export let get_env = (name: string): string => {
  if (inNodeJs) {
    // only available for Node.js
    return process.env[name];
  }
  if (typeof URLSearchParams != null) {
    return new URLSearchParams(location.search).get("env");
  }
  return null;
};

export let turn_keyword = (x: CrDataValue): CrDataKeyword => {
  if (typeof x === "string") {
    return kwd(x);
  }
  if (x instanceof CrDataKeyword) {
    return x;
  }
  if (x instanceof CrDataSymbol) {
    return kwd(x.value);
  }
  console.error(x);
  throw new Error("Unexpected data for keyword");
};

export let turn_symbol = (x: CrDataValue): CrDataKeyword => {
  if (typeof x === "string") {
    return new CrDataSymbol(x);
  }
  if (x instanceof CrDataSymbol) {
    return x;
  }
  if (x instanceof CrDataKeyword) {
    return new CrDataSymbol(x.value);
  }
  console.error(x);
  throw new Error("Unexpected data for symbol");
};

export let pr_str = (...args: CrDataValue[]): string => {
  return args.map((x) => toString(x, true)).join(" ");
};

/** helper function for println, js only */
export let printable = (...args: CrDataValue[]): string => {
  return args.map((x) => toString(x, false)).join(" ");
};

// time from app start
export let cpu_time = (): number => {
  if (inNodeJs) {
    // uptime returns in seconds
    return process.uptime() * 1000;
  }
  // returns in milliseconds
  return performance.now();
};

export let quit = (): void => {
  if (inNodeJs) {
    process.exit(1);
  } else {
    throw new Error("quit()");
  }
};

export let turn_string = (x: CrDataValue): string => {
  if (x == null) {
    return "";
  }
  if (typeof x === "string") {
    return x;
  }
  if (x instanceof CrDataKeyword) {
    return x.value;
  }
  if (x instanceof CrDataSymbol) {
    return x.value;
  }
  if (typeof x === "number") {
    return x.toString();
  }
  if (typeof x === "boolean") {
    return x.toString();
  }
  console.error(x);
  throw new Error("Unexpected data to turn string");
};

export let identical_QUES_ = (x: CrDataValue, y: CrDataValue): boolean => {
  return x === y;
};

export let starts_with_QUES_ = (xs: string, y: string): boolean => {
  return xs.startsWith(y);
};
export let ends_with_QUES_ = (xs: string, y: string): boolean => {
  return xs.endsWith(y);
};

export let blank_QUES_ = (x: string): boolean => {
  if (x == null) {
    return true;
  }
  if (typeof x === "string") {
    return x.trim() === "";
  } else {
    throw new Error("Expected a string");
  }
};

export let compare_string = (x: string, y: string) => {
  if (x < y) {
    return -1;
  }
  if (x > y) {
    return 1;
  }
  return 0;
};

export let arrayToList = (xs: Array<CrDataValue>): CrDataList => {
  return new CrDataList(xs ?? []);
};

export let listToArray = (xs: CrDataList): Array<CrDataValue> => {
  if (xs == null) {
    return null;
  }
  if (xs instanceof CrDataList) {
    return xs.toArray();
  } else {
    throw new Error("Expected list");
  }
};

export let number_QUES_ = (x: CrDataValue): boolean => {
  return typeof x === "number";
};
export let string_QUES_ = (x: CrDataValue): boolean => {
  return typeof x === "string";
};
export let bool_QUES_ = (x: CrDataValue): boolean => {
  return typeof x === "boolean";
};
export let nil_QUES_ = (x: CrDataValue): boolean => {
  return x == null;
};
export let keyword_QUES_ = (x: CrDataValue): boolean => {
  return x instanceof CrDataKeyword;
};
export let map_QUES_ = (x: CrDataValue): boolean => {
  return x instanceof CrDataMap;
};
export let list_QUES_ = (x: CrDataValue): boolean => {
  return x instanceof CrDataList;
};
export let set_QUES_ = (x: CrDataValue): boolean => {
  return x instanceof CrDataSet;
};
export let fn_QUES_ = (x: CrDataValue): boolean => {
  return typeof x === "function";
};
export let ref_QUES_ = (x: CrDataValue): boolean => {
  return x instanceof CrDataRef;
};
export let record_QUES_ = (x: CrDataValue): boolean => {
  return x instanceof CrDataRecord;
};
export let tuple_QUES_ = (x: CrDataValue): boolean => {
  return x instanceof CrDataTuple;
};

export let escape = (x: string) => JSON.stringify(x);

export let read_file = (path: string): string => {
  if (inNodeJs) {
    // TODO
    (globalThis as any)["__calcit_injections__"].read_file(path);
  } else {
    // no actual File API in browser
    return localStorage.get(path) ?? "";
  }
};
export let write_file = (path: string, content: string): void => {
  if (inNodeJs) {
    // TODO
    (globalThis as any)["__calcit_injections__"].write_file(path, content);
  } else {
    // no actual File API in browser
    localStorage.setItem(path, content);
  }
};

export let parse_cirru = (code: string): CrDataList => {
  return to_calcit_data(parse(code), true) as CrDataList;
};

export let parse_cirru_edn = (code: string) => {
  return extract_cirru_edn(parse(code)[0]);
};

/** return in seconds, like from Nim */
export let now_BANG_ = () => {
  return Date.now() / 1000;
};

/** return in seconds, like from Nim,
 * notice Nim version is slightly different
 */
export let parse_time = (text: string) => {
  return new Date(text).valueOf() / 1000;
};

export let format_to_lisp = (x: CrDataValue): string => {
  if (x == null) {
    return "nil";
  } else if (x instanceof CrDataSymbol) {
    return x.value;
  } else if (x instanceof CrDataList) {
    let chunk = "(";
    for (let item of x.items()) {
      if (chunk != "(") {
        chunk += " ";
      }
      chunk += format_to_lisp(item);
    }
    chunk += ")";
    return chunk;
  } else {
    return x.toString();
  }
};

/** for quickly creating js Array */
export let js_array = (...xs: CrDataValue[]): CrDataValue[] => {
  return xs;
};

export let _AND_js_object = (...xs: CrDataValue[]): Record<string, CrDataValue> => {
  if (xs.length % 2 !== 0) {
    throw new Error("&js-object expects even number of arguments");
  }
  var ret: Record<string, CrDataValue> = {}; // object
  let halfLength = xs.length >> 1;
  for (let idx = 0; idx < halfLength; idx++) {
    let k = xs[idx << 1];
    let v = xs[(idx << 1) + 1];
    if (typeof k === "string") {
      ret[k] = v;
    } else if (k instanceof CrDataKeyword) {
      ret[turn_string(k)] = v;
    } else {
      throw new Error("Invalid key for js Object");
    }
  }
  return ret;
};

/** notice, Nim version of format-time takes format */
export let format_time = (timeSecNumber: number, format?: string): string => {
  if (format != null) {
    console.error("format of calcit-js not implemented");
  }
  return new Date(timeSecNumber * 1000).toISOString();
};

export let _COL__COL_ = (a: CrDataValue, b: CrDataValue): CrDataTuple => {
  return new CrDataTuple(a, b);
};

// mutable place for core to register
let calcit_builtin_classes = {
  number: null as CrDataRecord,
  string: null as CrDataRecord,
  set: null as CrDataRecord,
  list: null as CrDataRecord,
  map: null as CrDataRecord,
  record: null as CrDataRecord,
};

// need to register code from outside
export let register_calcit_builtin_classes = (options: typeof calcit_builtin_classes) => {
  Object.assign(calcit_builtin_classes, options);
};

export function invoke_method(p: string) {
  return (obj: CrDataValue, ...args: CrDataValue[]) => {
    let klass: CrDataRecord;
    let value = obj;
    if (obj instanceof CrDataTuple) {
      if (obj.fst instanceof CrDataRecord) {
        klass = obj.fst;
      } else {
        throw new Error("Method invoking expected a record as class");
      }
    } else if (typeof obj === "number") {
      klass = calcit_builtin_classes.number;
    } else if (typeof obj === "string") {
      klass = calcit_builtin_classes.string;
    } else if (obj instanceof CrDataSet) {
      klass = calcit_builtin_classes.set;
    } else if (obj instanceof CrDataList) {
      klass = calcit_builtin_classes.list;
    } else if (obj instanceof CrDataRecord) {
      klass = calcit_builtin_classes.record;
    } else if (obj instanceof CrDataMap) {
      klass = calcit_builtin_classes.map;
    } else {
      return (obj as any)[p](...args); // trying to call JavaScript method
    }
    if (klass == null) {
      throw new Error("Cannot find class for this object for invoking");
    }
    let method = klass.get(p);
    if (typeof method === "function") {
      return method(value, ...args);
    } else {
      throw new Error("Method for invoking is not a function");
    }
  };
}

export let _AND_map_COL_to_list = (m: CrDataValue): CrDataList => {
  if (m instanceof CrDataMap) {
    let ys = [];
    for (let pair of m.pairs()) {
      ys.push(new CrDataList(pair));
    }
    return new CrDataList(ys);
  } else {
    throw new Error("&map:to-list expected a Map");
  }
};

export let _AND_compare = (a: CrDataValue, b: CrDataValue): number => {
  if (a < b) {
    return -1;
  } else if (a > b) {
    return 1;
  } else {
    return 0;
  }
};

// special procs have to be defined manually
export let reduce = foldl;
export let conj = append;

let unavailableProc = (...xs: []) => {
  console.warn("NOT available for calcit-js");
};

// not available for calcit-js
export let _AND_reset_gensym_index_BANG_ = unavailableProc;
export let dbt__GT_point = unavailableProc;
export let dbt_digits = unavailableProc;
export let dual_balanced_ternary = unavailableProc;
export let gensym = unavailableProc;
export let macroexpand = unavailableProc;
export let macroexpand_all = unavailableProc;
export let _AND_get_calcit_running_mode = unavailableProc;

// already handled in code emitter
export let raise = unavailableProc;
