//! This example showcases setting up a basic application and window, setting up some views to
//! work with autolayout, and some basic ways to handle colors.

use cacao::layout::{Layout, LayoutConstraint};
use cacao::input::{TextField, TextFieldDelegate};
use cacao::view::View;

use cacao::macos::{App, AppDelegate};
use cacao::macos::menu::{Menu, MenuItem};
use cacao::macos::window::{Window, WindowConfig, WindowDelegate};

struct BasicApp {
    window: Window<AppWindow>
}

impl AppDelegate for BasicApp {
    fn did_finish_launching(&self) {
        App::set_menu(vec![
            Menu::new("", vec![
                MenuItem::Services,
                MenuItem::Separator,
                MenuItem::Hide,
                MenuItem::HideOthers,
                MenuItem::ShowAll,
                MenuItem::Separator,
                MenuItem::Quit
            ]),

            Menu::new("File", vec![
                MenuItem::CloseWindow
            ]),

            Menu::new("Edit", vec![
                MenuItem::Undo,
                MenuItem::Redo,
                MenuItem::Separator,
                MenuItem::Cut,
                MenuItem::Copy,
                MenuItem::Paste,
                MenuItem::Separator,
                MenuItem::SelectAll
            ]),

            // Sidebar option is 11.0+ only.
            Menu::new("View", vec![
                MenuItem::EnterFullScreen
            ]),

            Menu::new("Window", vec![
                MenuItem::Minimize,
                MenuItem::Zoom,
                MenuItem::Separator,
                MenuItem::new("Bring All to Front")
            ]),

            Menu::new("Help", vec![])
        ]);

        App::activate();
        self.window.show();
    }
}

#[derive(Debug, Default)]
pub struct ConsoleLogger;

impl TextFieldDelegate for ConsoleLogger {
    const NAME: &'static str = "ConsoleLogger";

    fn text_should_begin_editing(&self, value: &str) -> bool {
        println!("Should begin with value: {}", value);
        true
    }

    fn text_did_change(&self, value: &str) {
        println!("Did change to: {}", value);
    }


    fn text_did_end_editing(&self, value: &str) {
        println!("Ended: {}", value);
    }
}

#[derive(Debug)]
struct AppWindow {
    input: TextField<ConsoleLogger>,
    content: View
}

impl AppWindow {
    pub fn new() -> Self {
        AppWindow {
            input: TextField::with(ConsoleLogger),
            content: View::new(),
        }
    }
}

impl WindowDelegate for AppWindow {
    const NAME: &'static str = "WindowDelegate";

    fn did_load(&mut self, window: Window) {
        window.set_title("Input Logger Example");
        window.set_minimum_content_size(300., 300.);

        self.content.add_subview(&self.input);
        window.set_content_view(&self.content);

        LayoutConstraint::activate(&[
            self.input.center_x.constraint_equal_to(&self.content.center_x),
            self.input.center_y.constraint_equal_to(&self.content.center_y),
            self.input.width.constraint_equal_to_constant(280.)
        ]);
    }
}

fn main() {
    App::new("com.test.window", BasicApp {
        window: Window::with(WindowConfig::default(), AppWindow::new())
    }).run();
}
