pub type _char = core::ffi::c_char;
pub type _uchar = core::ffi::c_uchar;
pub type _void = core::ffi::c_void;
pub type _str = CStr;
pub type _int = core::ffi::c_int;
pub type _long = core::ffi::c_long;
pub type _longlong = core::ffi::c_longlong;
pub type _ulong = core::ffi::c_ulong;
pub type _ulonglong = core::ffi::c_ulonglong;
pub type _ssize_t = isize;
pub type _size_t = usize;
pub type _int8_t = i8;
pub type _int16_t = i16;
pub type _int32_t = i32;
pub type _int64_t = i64;

#[macro_export]
macro_rules! str_to_char_ptr {
    ($string:expr) => {
        unsafe{$string.as_ptr() as *const _char}
    };
}

use _core::ascii;
use _core::cmp::Ordering;
use _core::ffi::c_char;
use _core::fmt::{self, Write};
use _core::ops;
use _core::slice;
use _core::slice::memchr;
use _core::str;

#[derive(Hash)]
struct CStr {
    inner: [c_char],
}
#[derive(Clone, PartialEq, Eq, Debug)]
struct FromBytesWithNulError {
    kind: FromBytesWithNulErrorKind,
}

#[derive(Clone, PartialEq, Eq, Debug)]
enum FromBytesWithNulErrorKind {
    InteriorNul(usize),
    NotNulTerminated,
}

impl FromBytesWithNulError {
    fn interior_nul(pos: usize) -> FromBytesWithNulError {
        FromBytesWithNulError { kind: FromBytesWithNulErrorKind::InteriorNul(pos) }
    }
    fn not_nul_terminated() -> FromBytesWithNulError {
        FromBytesWithNulError { kind: FromBytesWithNulErrorKind::NotNulTerminated }
    }

    pub fn __description(&self) -> &str {
        match self.kind {
            FromBytesWithNulErrorKind::InteriorNul(..) => {
                "data provided contains an interior nul byte"
            }
            FromBytesWithNulErrorKind::NotNulTerminated => "data provided is not nul terminated",
        }
    }
}
///
#[derive(Clone, PartialEq, Eq, Debug)]
struct FromBytesUntilNulError(());

impl fmt::Display for FromBytesUntilNulError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "data provided does not contain a nul")
    }
}

impl fmt::Debug for CStr {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "\"")?;
        for byte in self.to_bytes().iter().flat_map(|&b| ascii::escape_default(b)) {
            f.write_char(byte as char)?;
        }
        write!(f, "\"")
    }
}

impl Default for &CStr {
    fn default() -> Self {
        const SLICE: &[c_char] = &[0];
        unsafe { CStr::from_ptr(SLICE.as_ptr()) }
    }
}

impl fmt::Display for FromBytesWithNulError {
    #[allow(deprecated, deprecated_in_future)]
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.write_str(self.__description())?;
        if let FromBytesWithNulErrorKind::InteriorNul(pos) = self.kind {
            write!(f, " at byte pos {pos}")?;
        }
        Ok(())
    }
}

impl CStr {
    #[inline]
    #[must_use]
    pub unsafe fn from_ptr<'a>(ptr: *const c_char) -> &'a CStr {
        unsafe {
            extern "C" {
                fn strlen(s: *const c_char) -> usize;
            }
            let len = strlen(ptr);
            let ptr = ptr as *const u8;
            CStr::from_bytes_with_nul_unchecked(slice::from_raw_parts(ptr, len as usize + 1))
        }
    }

    pub fn from_bytes_until_nul(bytes: &[u8]) -> Result<&CStr, FromBytesUntilNulError> {
        let nul_pos = memchr::memchr(0, bytes);
        match nul_pos {
            Some(nul_pos) => {
                let subslice = &bytes[..nul_pos + 1];
                Ok(unsafe { CStr::from_bytes_with_nul_unchecked(subslice) })
            }
            None => Err(FromBytesUntilNulError(())),
        }
    }

    pub fn from_bytes_with_nul(bytes: &[u8]) -> Result<&Self, FromBytesWithNulError> {
        let nul_pos = memchr::memchr(0, bytes);
        match nul_pos {
            Some(nul_pos) if nul_pos + 1 == bytes.len() => {
                Ok(unsafe { Self::from_bytes_with_nul_unchecked(bytes) })
            }
            Some(nul_pos) => Err(FromBytesWithNulError::interior_nul(nul_pos)),
            None => Err(FromBytesWithNulError::not_nul_terminated()),
        }
    }

    #[inline]
    #[must_use]
    pub const unsafe fn from_bytes_with_nul_unchecked(bytes: &[u8]) -> &CStr {
        debug_assert!(!bytes.is_empty() && bytes[bytes.len() - 1] == 0);
        unsafe { Self::_from_bytes_with_nul_unchecked(bytes) }
    }

    #[inline]
    const unsafe fn _from_bytes_with_nul_unchecked(bytes: &[u8]) -> &CStr {
        unsafe { &*(bytes as *const [u8] as *const CStr) }
    }

    #[inline]
    #[must_use]
    pub const fn as_ptr(&self) -> *const c_char {
        self.inner.as_ptr()
    }

    #[inline]
    #[must_use = "this returns the result of the operation, \
                  without modifying the original"]
    pub fn to_bytes(&self) -> &[u8] {
        let bytes = self.to_bytes_with_nul();
        unsafe { bytes.get_unchecked(..bytes.len() - 1) }
    }
    #[inline]
    #[must_use = "this returns the result of the operation, \
                  without modifying the original"]
    pub fn to_bytes_with_nul(&self) -> &[u8] {
        unsafe { &*(&self.inner as *const [c_char] as *const [u8]) }
    }


    pub fn to_str(&self) -> Result<&str, str::Utf8Error> {
        str::from_utf8(self.to_bytes())
    }
}

impl PartialEq for CStr {
    fn eq(&self, other: &CStr) -> bool {
        self.to_bytes().eq(other.to_bytes())
    }
}
impl Eq for CStr {}
impl PartialOrd for CStr {
    fn partial_cmp(&self, other: &CStr) -> Option<Ordering> {
        self.to_bytes().partial_cmp(&other.to_bytes())
    }
}
impl Ord for CStr {
    fn cmp(&self, other: &CStr) -> Ordering {
        self.to_bytes().cmp(&other.to_bytes())
    }
}

impl ops::Index<ops::RangeFrom<usize>> for CStr {
    type Output = CStr;

    fn index(&self, index: ops::RangeFrom<usize>) -> &CStr {
        let bytes = self.to_bytes_with_nul();
        if index.start < bytes.len() {
            unsafe { CStr::from_bytes_with_nul_unchecked(&bytes[index.start..]) }
        } else {
            panic!(
                "index out of bounds: the len is {} but the index is {}",
                bytes.len(),
                index.start
            );
        }
    }
}

impl AsRef<CStr> for CStr {
    #[inline]
    fn as_ref(&self) -> &CStr {
        self
    }
}