use std::slice;
use crate::byte_buffer::ByteBuffer;
use crate::error::Result;

use cfg_if::cfg_if;

pub trait ByteBufferWrite {
	fn write_to_buffer(&self, buffer: &mut ByteBuffer) -> Result<()>;
	fn write_to_buffer_le(&self, buffer: &mut ByteBuffer) -> Result<()>;
	fn write_to_buffer_be(&self, buffer: &mut ByteBuffer) -> Result<()>;
}


impl ByteBufferWrite for u8 {
	#[inline]
	fn write_to_buffer(&self, buffer: &mut ByteBuffer) -> Result<()> {
		unsafe {
			buffer.write_slice(slice::from_raw_parts(self as *const u8, 1))?;
		}

		Ok(())
	}

	#[inline]
	fn write_to_buffer_le(&self, buffer: &mut ByteBuffer) -> Result<()> {
		self.write_to_buffer(buffer)
	}

	#[inline]
	fn write_to_buffer_be(&self, buffer: &mut ByteBuffer) -> Result<()> {
		self.write_to_buffer(buffer)
	}
}

impl ByteBufferWrite for &u8 {
	#[inline]
	fn write_to_buffer(&self, buffer: &mut ByteBuffer) -> Result<()> {
		unsafe {
			buffer.write_slice(slice::from_raw_parts(*self as *const u8, 1))?;
		}

		Ok(())
	}

	#[inline]
	fn write_to_buffer_le(&self, buffer: &mut ByteBuffer) -> Result<()> {
		self.write_to_buffer(buffer)
	}

	#[inline]
	fn write_to_buffer_be(&self, buffer: &mut ByteBuffer) -> Result<()> {
		self.write_to_buffer(buffer)
	}
}

impl ByteBufferWrite for i8 {
	#[inline]
	fn write_to_buffer(&self, buffer: &mut ByteBuffer) -> Result<()> {
		unsafe {
			buffer.write_slice(slice::from_raw_parts(self as *const i8 as *const u8, 1))?;
		}

		Ok(())
	}

	#[inline]
	fn write_to_buffer_le(&self, buffer: &mut ByteBuffer) -> Result<()> {
		self.write_to_buffer(buffer)
	}

	#[inline]
	fn write_to_buffer_be(&self, buffer: &mut ByteBuffer) -> Result<()> {
		self.write_to_buffer(buffer)
	}
}

impl ByteBufferWrite for &i8 {
	#[inline]
	fn write_to_buffer(&self, buffer: &mut ByteBuffer) -> Result<()> {
		unsafe {
			buffer.write_slice(slice::from_raw_parts(*self as *const i8 as *const u8, 1))?;
		}

		Ok(())
	}

	#[inline]
	fn write_to_buffer_le(&self, buffer: &mut ByteBuffer) -> Result<()> {
		self.write_to_buffer(buffer)
	}

	#[inline]
	fn write_to_buffer_be(&self, buffer: &mut ByteBuffer) -> Result<()> {
		self.write_to_buffer(buffer)
	}
}

macro_rules! impl_byte_buffer_write_types {
	($($type:ty),*) => {
		$(
			impl ByteBufferWrite for $type {
				#[inline]
				fn write_to_buffer(&self, buffer: &mut ByteBuffer) -> Result<()> {
					unsafe {
						buffer.write_slice(std::slice::from_raw_parts(self as *const $type as *const u8, std::mem::size_of::<$type>()))?;
					}

					Ok(())
				}

				#[inline]
				fn write_to_buffer_le(&self, buffer: &mut ByteBuffer) -> Result<()> {
					cfg_if! {
						if #[cfg(target_endian = "little")] {
							unsafe {
								buffer.write_slice(std::slice::from_raw_parts(self as *const $type as *const u8, std::mem::size_of::<$type>()))?;
							}
						} else {
							unsafe {
								let o = self.to_le_bytes();
								buffer.write_slice(std::slice::from_raw_parts(&o as *const u8, std::mem::size_of::<$type>()))?;
							}
						}
					}

					Ok(())
				}

				#[inline]
				fn write_to_buffer_be(&self, buffer: &mut ByteBuffer) -> Result<()> {
					cfg_if! {
						if #[cfg(target_endian = "big")] {
							unsafe {
								buffer.write_slice(std::slice::from_raw_parts(self as *const $type as *const u8, std::mem::size_of::<$type>()))?;
							}
						} else {
							unsafe {
								let o = self.to_be_bytes();
								buffer.write_slice(std::slice::from_raw_parts(&o as *const u8, std::mem::size_of::<$type>()))?;
							}
						}
					}

					Ok(())
				}
			}

			impl ByteBufferWrite for &$type {
				#[inline]
				fn write_to_buffer(&self, buffer: &mut ByteBuffer) -> Result<()> {
					unsafe {
						buffer.write_slice(std::slice::from_raw_parts(*self as *const $type as *const u8, std::mem::size_of::<$type>()))?;
					}

					Ok(())
				}

				#[inline]
				fn write_to_buffer_le(&self, buffer: &mut ByteBuffer) -> Result<()> {
					cfg_if! {
						if #[cfg(target_endian = "little")] {
							unsafe {
								buffer.write_slice(std::slice::from_raw_parts(*self as *const $type as *const u8, std::mem::size_of::<$type>()))?;
							}
						} else {
							unsafe {
								let o = self.to_le_bytes();
								buffer.write_slice(std::slice::from_raw_parts(&o as *const u8, std::mem::size_of::<$type>()))?;
							}
						}
					}

					Ok(())
				}

				#[inline]
				fn write_to_buffer_be(&self, buffer: &mut ByteBuffer) -> Result<()> {
					cfg_if! {
						if #[cfg(target_endian = "big")] {
							unsafe {
								buffer.write_slice(std::slice::from_raw_parts(*self as *const $type as *const u8, std::mem::size_of::<$type>()))?;
							}
						} else {
							unsafe {
								let o = self.to_be_bytes();
								buffer.write_slice(std::slice::from_raw_parts(&o as *const u8, std::mem::size_of::<$type>()))?;
							}
						}
					}

					Ok(())
				}
			}
		)*
	}
}

impl_byte_buffer_write_types!(u16, u32, u64, u128, usize, i16, i32, i64, i128, isize, f32, f64);
