use anyhow::{bail, Context, Result};
use base64::{STANDARD, STANDARD_NO_PAD, URL_SAFE, URL_SAFE_NO_PAD};
use std::{
    io::{self, Read},
    str::{self, FromStr},
};
use structopt::{clap::AppSettings::ColoredHelp, StructOpt};

/// Encode and decode various byte encodings.
///
/// # Examples
///
/// For example, if you have the hex string 4f33 and would like to base64-encode
/// the bytes [0x4f, 0x33] rather than the ASCII/UTF-8 characters '4', 'f',
/// etc., you could run the following:
///
/// ```shell
/// $ byt -r hex -R base64 4f33
/// TzM=
/// ```
///
/// To avoid padding the result and/or only use URL-safe characters, use the
/// --url-safe and --no-padding flags:
///
/// ```shell
/// $ byt -r hex -R base64 --url-safe --no-padding 4f33
/// TzM
/// ```
#[derive(StructOpt, Clone, Debug, PartialEq, PartialOrd)]
#[structopt(setting = ColoredHelp)]
struct Config {
    /// Treat the input as an encoded integer and decode it using this radix.
    #[structopt(short = "r", long = "radix", default_value = "utf8")]
    input_radix: Radix,

    /// The desired radix of the output string.
    #[structopt(short = "R", long, default_value = "64")]
    output_radix: Radix,

    /// [base64-only] Use URL-safe encoding.
    #[structopt(short, long)]
    url_safe: bool,

    /// [base64-only] Don't pad the output.
    #[structopt(short, long)]
    no_padding: bool,

    /// Data to (de|en)code. Will read from STDIN if not passed as an argument.
    data: Option<String>,
}

#[derive(Clone, Debug, PartialEq, PartialOrd)]
enum Radix {
    Binary,
    Base16,
    Base64,
    Utf8,
}

impl FromStr for Radix {
    type Err = anyhow::Error;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        Ok(match s {
            "binary" | "bin" | "2" => Self::Binary,
            "base16" | "hex" | "16" => Self::Base16,
            "base64" | "64" => Self::Base64,
            "utf8" => Self::Utf8,
            _ => bail!("unrecognized radix; should be 2, 16, 64 or utf8"),
        })
    }
}

fn main() -> Result<()> {
    let config = Config::from_args();

    let encode_config = if config.url_safe {
        if config.no_padding {
            URL_SAFE_NO_PAD
        } else {
            URL_SAFE
        }
    } else if config.no_padding {
        STANDARD_NO_PAD
    } else {
        STANDARD
    };

    let input = if let Some(data) = config.data {
        data
    } else {
        let mut data = vec![];
        io::stdin()
            .lock()
            .read_to_end(&mut data)
            .context("unable to read stdin")?;

        str::from_utf8(&data)
            .context("unable to parse stdin into utf8 string")?
            .to_string()
    };

    let input = match config.input_radix {
        Radix::Binary => bail!("parsing binary input is not supported"),
        Radix::Base16 => hex::decode(input.trim()).context("unable to decode input from base16")?,
        Radix::Base64 => base64::decode_config(input.trim(), encode_config)
            .context("unable to decode input from base64")?,
        Radix::Utf8 => input.as_bytes().to_vec(),
    };

    let output = match config.output_radix {
        Radix::Binary => input
            .iter()
            .map(|b| format!("{:08b}", b))
            .collect::<Vec<String>>()
            .join(" "),
        Radix::Base16 => hex::encode(input),
        Radix::Base64 => base64::encode_config(input, encode_config),
        Radix::Utf8 => str::from_utf8(&input)
            .context("unable to encode bytes as utf8")?
            .to_string(),
    };

    print!("{}", output);

    Ok(())
}
