use std::collections::HashMap;
use std::fs;
use serde_json;
use std::process::{Command, Stdio};
use std::io::prelude::*;
use byggis::ByggisErrors;
use regex::Regex;
use std::io;
use crossterm::style::*;
use byggis::SupportedLanguages;

// NOTE: Should probably split this into multiple functions for easier reading and stuff
pub fn run_tests() -> Result<(), ByggisErrors> {

    // get .byggis file to read tests from
    let dot_byggis = match fs::File::open("./.byggis") {
        Ok(n) => n,
        Err(_) => { return Err(ByggisErrors::ByggisFileNotFound); },
    };

    // reads and error handles tests from .byggis file
    // TODO: rewrite how the hashmap is designed to be something akin to:
    //      {
    //           test_inputs : {
    //               []
    //           },
    //           test_outputs : {
    //                []
    //           },
    //       }
    // to let this serialize well with serde we should prolly serialize it into 
    // a custom struct, however this is not necesary until we work on commiting
    // The DotByggis struct in lib.rs can be used for this 

    let tests: HashMap<String, String> = match serde_json::from_reader(dot_byggis) {
        Ok(n) => n,
        Err(_) => { return Err(ByggisErrors::TestsNotFound); },
    };


    // use regex to put the files in a vector for easy access
    let re = Regex::new(r"\./main\..{1,5}").unwrap();
    let f_vec: Vec<String> = fs::read_dir("./")
        .unwrap()
        .map(|x| x.unwrap()
            .path()
            .display()
            .to_string())
        .filter(|x| re.is_match(x))
        .collect();


    // error handling for if the folder is empty, e.g no files found
    if f_vec.is_empty() {
        return Err(ByggisErrors::MainNotFound);
    }


    // get and parse the input from the user
    let mut num: i32 = 1;
    if f_vec.len() > 1 {
        // handles more than one main file
        println!("  {}: Detected more than one main file...",
            "Note".blue());
        println!("   Select main file to use:");


        // prints out the files in a nice manner
        for (i, f) in f_vec.iter().enumerate() {
            println!("     {}: {}",
                i+1,
                f.to_string().bold());
        }


        // read from stdin to n to be used as a option in selection process
        let mut n = String::new();
        io::stdin().read_line(&mut n).expect("Could not read from stdin");
        n.pop();


        // parse the input from n/stdin into a clean integer
        num = n.parse().unwrap_or_else(|_| {
            println!("    {}: Could not convert to int, defaulting to first option.",
                "Error".red());
            1
        });


        // error checking operation, defaults to first option
        if num > f_vec.len() as i32 {
            println!("    {}: Number not an option, defaulting to first option",
                "Error".red());
            num = 1;
        }
    }

    // gets the file name from the vector of names based on the inputed index
    let main_file: &String = &f_vec[(num-1) as usize];

    let language = SupportedLanguages::from_string(main_file.split(".").last().unwrap().to_string()).unwrap();

    // get the used language and compile/setup for running the file
    match language {
        SupportedLanguages::Python => {},
        SupportedLanguages::Rust => {
            // create a process for compiling rust file and check output
            let p = Command::new("rustc")
                .arg("-A")
                .arg("warnings")
                .arg("main.rs")
                .stdin( Stdio::piped())
                .stdout(Stdio::piped())
                .stderr(Stdio::piped())
                .spawn()
                .unwrap();

            let o = &p.wait_with_output();
            let stderr = &String::from_utf8_lossy(&o.as_ref().unwrap().stderr);

            if stderr != "" {
                return Err(ByggisErrors::CompileTimeError(stderr.trim().to_string()))
            }
        },
        SupportedLanguages::Java => {
            let p = Command::new("javac")
                .arg("main.java")
                .stdin( Stdio::piped())
                .stdout(Stdio::piped())
                .stderr(Stdio::piped())
                .spawn()
                .unwrap();

            let o = &p.wait_with_output();
            let stderr = &String::from_utf8_lossy(&o.as_ref().unwrap().stderr);

            if stderr != "" {
                return Err(ByggisErrors::CompileTimeError(stderr.trim().to_string()))
            }
        },
        SupportedLanguages::Haskell => {
            let p = Command::new("ghc")
                .arg("main.hs")
                .stdin( Stdio::piped())
                .stdout(Stdio::piped())
                .stderr(Stdio::piped())
                .spawn()
                .unwrap();

            let o = &p.wait_with_output();
            let stderr = &String::from_utf8_lossy(&o.as_ref().unwrap().stderr);

            if stderr != "" {
                return Err(ByggisErrors::CompileTimeError(stderr.trim().to_string()))
            }
        }
    };


    // run the file against the tests
    for (s_input, s_output) in tests {

        // spawn process and execute file
        let mut p;
        match language {
            SupportedLanguages::Rust => {
                p = Command::new("./main")
                    .stdin( Stdio::piped())
                    .stdout(Stdio::piped())
                    .stderr(Stdio::piped())
                    .spawn()
                    .unwrap();
            },
            SupportedLanguages::Python => {
                p = Command::new("python")
                    .arg("main.py")
                    .stdin( Stdio::piped())
                    .stdout(Stdio::piped())
                    .stderr(Stdio::piped())
                    .spawn()
                    .unwrap();
            }, 
            SupportedLanguages::Java => {
                p = Command::new("java")
                    .arg("main.class")
                    .stdin( Stdio::piped())
                    .stdout(Stdio::piped())
                    .stderr(Stdio::piped())
                    .spawn()
                    .unwrap();
            }, 
            SupportedLanguages::Haskell => {
                p = Command::new("./main")
                    .stdin( Stdio::piped())
                    .stdout(Stdio::piped())
                    .stderr(Stdio::piped())
                    .spawn()
                    .unwrap();
            }
        }

        // feed the process the input from the file
        p.stdin
            .as_mut()
            .unwrap()
            .write(s_input.as_bytes())
            .unwrap();

        println!("   Test case:");
        for s in s_input.split("\n") {
            println!("     {}", s.trim().italic());
        }

        let o = &p.wait_with_output();

        // print out the test results
        if String::from_utf8_lossy(&o.as_ref().unwrap().stdout) == s_output {
            println!("    Test: {}\n", "ok".green());
        } else {
            println!("    Test: {}", "failed".red());

            // handle runtime errors and pretty print them
            //  NOTE: This does not get handled by main.rs since its a 
            //        recoverable error, e.g we still want the program to finish 
            //        safely after this happens
            if String::from_utf8_lossy(&o.as_ref().unwrap().stderr).trim() != "" {

                println!("     Error:");

                for l in String::from_utf8_lossy(&o.as_ref().unwrap().stderr).trim().split("\n") {
                    println!("      {}", l.bold());
                }

                println!();
            } else {
                // prints output of the test
                println!("     Output: \n{}",
                    String::from_utf8_lossy(&o
                        .as_ref()
                        .unwrap()
                        .stdout).trim().italic());
                println!("");
            }
        }
    }

    Ok(())
}
