use assert_cmd::Command;
use predicates::str::{contains, is_empty};

/// Convenience method for getting the command
fn get_command() -> Command {
    Command::cargo_bin("buzzec-strip-dag-node").expect("Could not find command")
}

/// Test no input behavior
#[test]
fn null_input_test() {
    get_command()
        .write_stdin("")
        .assert()
        .success()
        .stdout("\n")
        .stderr(is_empty());
}

/// Tests stdin reading
#[test]
fn stdin_test() {
    get_command()
        .arg("--strip")
        .arg("c")
        .write_stdin("a-b,b-c,c-d")
        .assert()
        .success()
        .stdout("a-b,b-d\n")
        .stderr(is_empty());

    get_command()
        .arg("-s")
        .arg("c")
        .write_stdin("a-b,b-c,c-d")
        .assert()
        .success()
        .stdout("a-b,b-d\n")
        .stderr(is_empty());
}

/// Tests passing input through `--in`
#[test]
fn arg_test() {
    get_command()
        .arg("--in")
        .arg("a-b,b-c,c-d")
        .arg("--strip")
        .arg("c")
        .assert()
        .success()
        .stdout("a-b,b-d\n")
        .stderr(is_empty());

    get_command()
        .arg("-i")
        .arg("a-b,b-c,c-d")
        .arg("-s")
        .arg("c")
        .assert()
        .success()
        .stdout("a-b,b-d\n")
        .stderr(is_empty());
}

/// Tests having no strip passed
#[test]
fn no_strip_test() {
    get_command()
        .write_stdin("a-b,b-c,c-d")
        .assert()
        .success()
        .stdout("a-b,b-c,c-d\n")
        .stderr(is_empty());
}

/// Tests what happens if passed a bad dag
#[test]
fn bad_dag_test() {
    get_command()
        .write_stdin("a-b,b-c,c-a")
        .assert()
        .failure()
        .stdout(is_empty())
        .stderr(contains("Given DAG is invalid"));
}

/// Tests passing multiple of the same edge dag
#[test]
fn multiple_same_edge_test() {
    get_command()
        .write_stdin("a-b,b-c,b-c,c-d")
        .assert()
        .success()
        .stdout("a-b,b-c,c-d\n")
        .stderr(is_empty());
}

/// No guarantees about what the output is when bad input is given and verification is off.
#[test]
fn no_verify_test() {
    get_command()
        .write_stdin("a-b,b-c,c-a")
        .arg("--no-verify")
        .assert()
        .success();
}

/// Tests multiple strips in the same command
#[test]
fn multi_strip_test() {
    get_command()
        .write_stdin("a-b,b-c,b-d,c-d")
        .arg("-s")
        .arg("b")
        .arg("-s")
        .arg("d")
        .assert()
        .success()
        .stdout("a-c\n")
        .stderr(is_empty());
}

/// Tests stripping a node that is not present
#[test]
fn not_present_strip() {
    get_command()
        .write_stdin("a-b,b-c,b-d,c-d")
        .arg("-s")
        .arg("e")
        .assert()
        .success()
        .stdout("a-b,b-c,b-d,c-d\n")
        .stderr(is_empty());
}

/// Tests the case that a node with a large number of connections is removed.
#[test]
fn large_connection_strip() {
    get_command()
        .write_stdin("a-d,b-d,c-d,d-e,d-f")
        .arg("-s")
        .arg("d")
        .assert()
        .success()
        .stdout("a-e,a-f,b-e,b-f,c-e,c-f\n")
        .stderr(is_empty());
}
