use structopt::StructOpt;

#[derive(StructOpt)]
#[structopt(
    name = "buzzec-strip-dag-node",
    about = "Strips out a dag node. Defaults to stdin, use --in to input a command line argument. If skip and no-verify are not present will verify and format the given DAG."
)]
pub struct Opt {
    /// Strips a given node from the DAG. If the node is not present the DAG will not change.
    #[structopt(short = "s", long = "strip")]
    pub strip: Option<Vec<String>>,
    /// Optional argument for evaluating from a command line argument rather than stdin.
    #[structopt(short = "i", long = "in")]
    pub in_: Option<String>,
    /// Disables verification. No guarantees on the output if given a bad DST.
    #[structopt(long = "no-verify")]
    pub no_verify: bool,
}

#[cfg(test)]
mod test {
    use structopt::StructOpt;

    use super::Opt;

    #[test]
    fn opt_test() {
        let opt: Opt = Opt::from_iter_safe(vec![""].into_iter()).expect("could not parse");
        assert!(opt.strip.is_none());
        assert!(opt.in_.is_none());
        assert!(!opt.no_verify);

        let opt: Opt =
            Opt::from_iter_safe(vec!["", "-s", "a"].into_iter()).expect("could not parse");
        assert_eq!(opt.strip.expect("No strip"), vec!["a".to_string()]);
        assert!(opt.in_.is_none());
        assert!(!opt.no_verify);

        let opt: Opt =
            Opt::from_iter_safe(vec!["", "-s", "a", "--no-verify", "-s", "d"].into_iter())
                .expect("could not parse");
        assert_eq!(
            opt.strip.expect("No strip"),
            vec!["a".to_string(), "d".to_string()]
        );
        assert!(opt.in_.is_none());
        assert!(opt.no_verify);
        let opt: Opt = Opt::from_iter_safe(
            vec![
                "",
                "-s",
                "a",
                "--no-verify",
                "--strip",
                "d",
                "--in",
                "a-b,a-c",
            ]
            .into_iter(),
        )
        .expect("could not parse");
        assert_eq!(
            opt.strip.expect("No strip"),
            vec!["a".to_string(), "d".to_string()]
        );
        assert_eq!(opt.in_, Some("a-b,a-c".to_string()));
        assert!(opt.no_verify);
    }
}
