use std::collections::{HashMap, HashSet, VecDeque};
use std::error::Error;
use std::fmt;
use std::fmt::{Debug, Display, Formatter};
use std::iter::{once, FromIterator};

use bimap::BiMap;
use slotmap::{new_key_type, SlotMap};

new_key_type! { struct NodeKey; }

/// A Dag from user input
#[derive(Debug)]
pub struct Dag {
    node_name_map: BiMap<String, NodeKey>,
    nodes: SlotMap<NodeKey, Node>,
}
impl Dag {
    /// Creates a DAG from a given string, panics if formatted improperly
    pub fn parse_str(parse: impl AsRef<str>) -> Result<Self, DagParseError> {
        parse
            .as_ref()
            .chars()
            .filter(|c| !c.is_whitespace())
            .collect::<String>()
            .split(',')
            .filter_map(|string| {
                if string.is_empty() {
                    return None;
                }
                let mut split = string.split('-');
                let first = split.next()?;
                let second = split.next().unwrap_or("");
                if let Some(third) = split.next() {
                    Some(Err(DagParseError::BadArgumentPart {
                        first: first.to_string(),
                        second: second.to_string(),
                        third: third.to_string(),
                    }))
                } else {
                    Some(Ok((first, second)))
                }
            })
            .collect()
    }

    /// Prints the DAG in the proper format.
    pub fn print_alphabetically(self) -> String {
        let node_name_map = self.node_name_map;
        let mut edges: Vec<String> = self
            .nodes
            .into_iter()
            .map(|(key, node)| (node_name_map.get_by_right(&key).unwrap(), node))
            .map(
                |(name, node): (&String, Node)| -> Box<dyn Iterator<Item = String>> {
                    let name = name;
                    if !node.children.is_empty() || !node.parents.is_empty() {
                        Box::new(
                            node.children
                                .into_iter()
                                .map(|key| node_name_map.get_by_right(&key).unwrap())
                                .map(move |child_name| format!("{}-{}", name, child_name)),
                        )
                    } else {
                        Box::new(once(name.clone()))
                    }
                },
            )
            .flatten()
            .collect();
        edges.sort_unstable();
        let mut out = String::with_capacity(edges.len() * 2);
        let mut edges = edges.into_iter();
        if let Some(edge) = edges.next() {
            out += edge.as_str();
        }
        for edge in edges {
            out += ",";
            out += edge.as_str();
        }
        out
    }

    /// Verifies that this is a valid DAG.
    pub fn verify(&self) -> Result<(), DagVerifyError> {
        // Catches when there are no top nodes and everything is a cycle.
        let mut top_node_count = 0usize;

        self.nodes
            .iter()
            .filter(|(_, node)| node.parents.is_empty())
            .map(|(index, _)| {
                top_node_count += 1;
                let mut parents = HashSet::new();
                let mut operations = VecDeque::new();
                operations.push_front(Operation::Scan(index));
                while let Some(operation) = operations.pop_front() {
                    match operation {
                        Operation::Scan(scan_index) => {
                            if !parents.insert(scan_index) {
                                return Err(DagVerifyError::CycleDetected(Some(
                                    parents
                                        .into_iter()
                                        .map(|key| {
                                            self.node_name_map.get_by_right(&key).unwrap().clone()
                                        })
                                        .collect(),
                                )));
                            }
                            operations.push_front(Operation::Remove(scan_index));
                            for &child in self.nodes[scan_index].children.iter() {
                                operations.push_front(Operation::Scan(child));
                            }
                        }
                        Operation::Remove(remove_index) => assert!(parents.remove(&remove_index)),
                    }
                }
                Ok(())
            })
            .collect::<Result<Vec<()>, DagVerifyError>>()?;

        if top_node_count == 0 && !self.node_name_map.is_empty() {
            Err(DagVerifyError::CycleDetected(None))
        } else {
            Ok(())
        }
    }

    /// Removes a node from this DAG.
    pub fn remove_node(&mut self, node_name: impl AsRef<str>) {
        let (_, key) = match self.node_name_map.remove_by_left(node_name.as_ref()) {
            None => return,
            Some(val) => val,
        };

        let removed_node = self.nodes.remove(key).unwrap();
        for &parent in removed_node.parents.iter() {
            let parent = self.nodes.get_mut(parent).unwrap();
            assert!(parent.children.remove(&key));
            for &child in removed_node.children.iter() {
                parent.children.insert(child);
            }
        }
        for child in removed_node.children {
            let child = self.nodes.get_mut(child).unwrap();
            assert!(child.parents.remove(&key));
            for &parent in removed_node.parents.iter() {
                child.parents.insert(parent);
            }
        }
    }
}
/// Second tuple member is empty string if single node.
impl<T, U> FromIterator<(T, U)> for Dag
where
    T: Into<String>,
    U: Into<String>,
{
    fn from_iter<I: IntoIterator<Item = (T, U)>>(iter: I) -> Self {
        let mut node_name_map = BiMap::new();
        let mut nodes: SlotMap<NodeKey, Node> = SlotMap::with_key();
        for (parent, child) in iter {
            let child_string = child.into();
            if child_string.is_empty() {
                get_or_insert_key(&mut node_name_map, &mut nodes, parent.into());
            } else {
                let parent_index = get_or_insert_key(&mut node_name_map, &mut nodes, parent.into());
                let child_index = get_or_insert_key(&mut node_name_map, &mut nodes, child_string);
                nodes[parent_index].children.insert(child_index);
                nodes[child_index].parents.insert(parent_index);
            }
        }
        Self {
            node_name_map,
            nodes,
        }
    }
}
impl PartialEq for Dag {
    fn eq(&self, other: &Self) -> bool {
        if self.node_name_map.len() != other.node_name_map.len() {
            return false;
        }
        let mut self_to_other = HashMap::new();
        for (name, &self_key) in self.node_name_map.iter() {
            let other_key = match other.node_name_map.get_by_left(name) {
                None => return false,
                Some(&key) => key,
            };
            self_to_other.insert(self_key, other_key);
        }
        for (key, self_node) in self.nodes.iter() {
            let other_node = other.nodes.get(self_to_other[&key]).unwrap();
            for parent in self_node.parents.iter() {
                if !other_node.parents.contains(&self_to_other[parent]) {
                    return false;
                }
            }
            for child in self_node.children.iter() {
                if !other_node.children.contains(&self_to_other[child]) {
                    return false;
                }
            }
        }
        true
    }
}
impl Eq for Dag {}

#[derive(Debug, Default)]
pub struct Node {
    parents: HashSet<NodeKey>,
    children: HashSet<NodeKey>,
}

#[derive(Debug)]
pub enum DagVerifyError {
    /// Member is nodes involved. This may or may not be available information.
    CycleDetected(Option<Vec<String>>),
}
impl Display for DagVerifyError {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        <Self as Debug>::fmt(self, f)
    }
}
impl Error for DagVerifyError {}

#[derive(Debug)]
pub enum DagParseError {
    /// One set has more than one `-` in a single `,` block
    BadArgumentPart {
        first: String,
        second: String,
        third: String,
    },
}
impl Display for DagParseError {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        <Self as Debug>::fmt(self, f)
    }
}
impl Error for DagParseError {}

#[derive(Copy, Clone, Debug)]
enum Operation {
    /// Add children and remove self to list
    Scan(NodeKey),
    /// Remove from parent list
    Remove(NodeKey),
}

/// Gets the key of `node_name` or inserts and returns it
fn get_or_insert_key(
    node_name_map: &mut BiMap<String, NodeKey>,
    nodes: &mut SlotMap<NodeKey, Node>,
    node_name: String,
) -> NodeKey {
    node_name_map
        .get_by_left(&node_name)
        .cloned()
        .unwrap_or_else(|| {
            let out = nodes.insert(Default::default());
            node_name_map.insert(node_name, out);
            out
        })
}

#[cfg(test)]
mod test {
    use crate::dag::Dag;

    #[test]
    fn verify_test() {
        let dag: Dag = vec![("a", "b"), ("b", "c"), ("b", "d")]
            .into_iter()
            .collect();
        assert!(dag.verify().is_ok());

        let dag: Dag = vec![("a", "b"), ("b", "c"), ("c", "a")]
            .into_iter()
            .collect();
        assert!(dag.verify().is_err());

        let dag: Dag = vec![("a", "b"), ("b", "c"), ("c", "b")]
            .into_iter()
            .collect();
        let result = dag.verify();
        assert!(result.is_err());
    }

    #[test]
    fn remove_test() {
        let mut dag: Dag = vec![("a", "b"), ("b", "c"), ("b", "d")]
            .into_iter()
            .collect();
        dag.remove_node("b");
        assert_eq!(dag, vec![("a", "c"), ("a", "d"),].into_iter().collect());

        let mut dag: Dag = vec![("a", "b"), ("c", ""), ("b", "d")]
            .into_iter()
            .collect();
        dag.remove_node("c");
        assert_eq!(dag, vec![("a", "b"), ("b", "d"),].into_iter().collect());
    }

    #[test]
    fn print_alphabetically_test() {
        let dag: Dag = vec![("a", "b"), ("b", "c"), ("b", "d")]
            .into_iter()
            .collect();
        assert_eq!(dag.print_alphabetically(), "a-b,b-c,b-d");

        let dag: Dag = vec![("a", "b"), ("c", ""), ("b", "d")]
            .into_iter()
            .collect();
        assert_eq!(dag.print_alphabetically(), "a-b,b-d,c");

        let dag: Dag = vec![("x", "a"), ("a", "b"), ("c", ""), ("b", "d")]
            .into_iter()
            .collect();
        assert_eq!(dag.print_alphabetically(), "a-b,b-d,c,x-a");
    }

    #[test]
    fn parse_str_test() {
        assert_eq!(
            Dag::parse_str("a-b,b-c,b-d").expect("Could not parse"),
            vec![("a", "b"), ("b", "c"), ("b", "d"),]
                .into_iter()
                .collect()
        );

        let dag: Dag = vec![("x", "a"), ("a", "b"), ("c", ""), ("b", "d")]
            .into_iter()
            .collect();
        assert_eq!(
            dag,
            Dag::parse_str("x-a,a-b,b-d,c").expect("Could not parse")
        );
    }
}
