// Buttplug Rust Source Code File - See https://buttplug.io for more info.
//
// Copyright 2016-2021 Nonpolynomial Labs LLC. All rights reserved.
//
// Licensed under the BSD 3-Clause license. See LICENSE file in the project root
// for full license information.

//! Device specific identification and protocol implementations.

use super::protocol::{
  add_to_protocol_map,
  get_default_protocol_map,
  ButtplugProtocol,
  TryCreateProtocolFunc,
};
use crate::{
  core::{
    errors::{ButtplugDeviceError, ButtplugError},
    messages::{ButtplugDeviceMessageType, DeviceMessageAttributes, DeviceMessageAttributesMap},
  },
  device::Endpoint,
};
use dashmap::DashMap;
use serde::{Deserialize, Serialize};
use std::{
  collections::{HashMap, HashSet},
  sync::Arc,
};
use uuid::Uuid;

// Note: There's a ton of extra structs in here just to deserialize the json
// file. Just leave them and build extras (for instance,
// DeviceProtocolConfiguration) if needed elsewhere in the codebase. It's not
// gonna hurt anything and making a ton of serde attributes is just going to get
// confusing (see the messages impl).

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct BluetoothLESpecifier {
  pub names: HashSet<String>,
  #[serde(default, rename = "advertised-services")]
  pub advertised_services: HashSet<Uuid>,
  // Set of services that we may have gotten as part of the advertisement.
  pub services: HashMap<Uuid, HashMap<Endpoint, Uuid>>,
}

impl PartialEq for BluetoothLESpecifier {
  fn eq(&self, other: &Self) -> bool {
    // If names or advertised services are found, use those automatically.
    if self.names.intersection(&other.names).count() > 0 {
      return true;
    }
    if self
      .advertised_services
      .intersection(&other.advertised_services)
      .count()
      > 0
    {
      return true;
    }
    // Otherwise, try wildcarded names.
    for name in &self.names {
      for other_name in &other.names {
        let compare_name: &String;
        let mut wildcard: String;
        if name.ends_with('*') {
          wildcard = name.clone();
          compare_name = other_name;
        } else if other_name.ends_with('*') {
          wildcard = other_name.clone();
          compare_name = name;
        } else {
          continue;
        }
        // Remove asterisk from the end of the wildcard
        wildcard.pop();
        if compare_name.starts_with(&wildcard) {
          return true;
        }
      }
    }
    false
  }
}

impl BluetoothLESpecifier {
  pub fn new_from_device(name: &str, advertised_services: &[Uuid]) -> BluetoothLESpecifier {
    let mut name_set = HashSet::new();
    name_set.insert(name.to_string());
    let service_set = HashSet::from_iter(advertised_services.iter().copied());
    BluetoothLESpecifier {
      names: name_set,
      advertised_services: service_set,
      services: HashMap::new(),
    }
  }

  pub fn merge(&mut self, other: BluetoothLESpecifier) {
    // Add any new names.
    self.names = self.names.union(&other.names).cloned().collect();
    // Add new services, overwrite matching services.
    self.services.extend(other.services);
  }
}

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct LovenseConnectServiceSpecifier {
  // Needed for proper deserialization, but clippy will complain.
  #[allow(dead_code)]
  exists: bool,
}

impl Default for LovenseConnectServiceSpecifier {
  fn default() -> Self {
    Self { exists: true }
  }
}

impl PartialEq for LovenseConnectServiceSpecifier {
  fn eq(&self, _other: &Self) -> bool {
    true
  }
}

#[derive(Serialize, Deserialize, Debug, Clone, Copy)]
pub struct XInputSpecifier {
  // Needed for deserialziation but unused.
  #[allow(dead_code)]
  exists: bool,
}

impl Default for XInputSpecifier {
  fn default() -> Self {
    Self { exists: true }
  }
}

impl PartialEq for XInputSpecifier {
  fn eq(&self, _other: &Self) -> bool {
    true
  }
}

#[derive(Serialize, Deserialize, Debug, PartialEq, Clone, Copy)]
pub struct HIDSpecifier {
  #[serde(rename = "vendor-id")]
  vendor_id: u16,
  #[serde(rename = "product-id")]
  product_id: u16,
}

#[derive(Serialize, Deserialize, Debug, Clone, Default)]
pub struct SerialSpecifier {
  #[serde(rename = "baud-rate")]
  pub baud_rate: u32,
  #[serde(rename = "data-bits")]
  pub data_bits: u8,
  #[serde(rename = "stop-bits")]
  pub stop_bits: u8,
  pub parity: char,
  pub port: String,
}

impl SerialSpecifier {
  pub fn new_from_name(port: &str) -> Self {
    SerialSpecifier {
      port: port.to_owned(),
      ..Default::default()
    }
  }
}

impl PartialEq for SerialSpecifier {
  fn eq(&self, other: &Self) -> bool {
    self.port == other.port
  }
}

#[derive(Serialize, Deserialize, Debug, PartialEq, Clone, Copy)]
pub struct USBSpecifier {
  #[serde(rename = "vendor-id")]
  vendor_id: u16,
  #[serde(rename = "product-id")]
  product_id: u16,
}

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct WebsocketSpecifier {
  pub names: HashSet<String>,
}

impl WebsocketSpecifier {
  pub fn merge(&mut self, other: WebsocketSpecifier) {
    // Just add the new identifier names
    self.names.extend(other.names);
  }
}

impl PartialEq for WebsocketSpecifier {
  fn eq(&self, other: &Self) -> bool {
    if self.names.intersection(&other.names).count() > 0 {
      return true;
    }
    false
  }
}

impl WebsocketSpecifier {
  pub fn new(name: &str) -> WebsocketSpecifier {
    let mut set = HashSet::new();
    set.insert(name.to_string());
    WebsocketSpecifier { names: set }
  }
}

#[derive(Serialize, Deserialize, Debug, PartialEq, Clone)]
pub enum DeviceSpecifier {
  BluetoothLE(BluetoothLESpecifier),
  HID(HIDSpecifier),
  USB(USBSpecifier),
  Serial(SerialSpecifier),
  XInput(XInputSpecifier),
  LovenseConnectService(LovenseConnectServiceSpecifier),
  Websocket(WebsocketSpecifier),
}

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct ProtocolAttributes {
  identifier: Option<Vec<String>>,
  name: Option<HashMap<String, String>>,
  messages: Option<DeviceMessageAttributesMap>,
}

#[derive(Deserialize, Serialize, Debug, Clone, Default)]
pub struct ProtocolDefinition {
  // Can't get serde flatten specifiers into a String/DeviceSpecifier map, so
  // they're kept separate here, and we return them in get_specifiers(). Feels
  // very clumsy, but we really don't do this a bunch during a session.
  pub usb: Option<Vec<USBSpecifier>>,
  pub btle: Option<BluetoothLESpecifier>,
  pub serial: Option<Vec<SerialSpecifier>>,
  pub hid: Option<Vec<HIDSpecifier>>,
  pub xinput: Option<XInputSpecifier>,
  pub websocket: Option<WebsocketSpecifier>,
  #[serde(rename = "lovense-connect-service")]
  pub lovense_connect_service: Option<LovenseConnectServiceSpecifier>,
  pub defaults: Option<ProtocolAttributes>,
  #[serde(default)]
  pub configurations: Vec<ProtocolAttributes>,
}

fn option_some_eq<T>(a: &Option<T>, b: &T) -> bool
where
  T: PartialEq,
{
  a.as_ref().map_or(false, |x| x == b)
}

fn option_some_eq_vec<T>(a_opt: &Option<Vec<T>>, b: &T) -> bool
where
  T: PartialEq,
{
  a_opt.as_ref().map_or(false, |a_vec| a_vec.contains(b))
}

impl PartialEq<DeviceSpecifier> for ProtocolDefinition {
  fn eq(&self, other: &DeviceSpecifier) -> bool {
    // TODO This seems like a really gross way to do this?
    match other {
      DeviceSpecifier::USB(other_usb) => option_some_eq_vec(&self.usb, other_usb),
      DeviceSpecifier::Serial(other_serial) => option_some_eq_vec(&self.serial, other_serial),
      DeviceSpecifier::BluetoothLE(other_btle) => option_some_eq(&self.btle, other_btle),
      DeviceSpecifier::HID(other_hid) => option_some_eq_vec(&self.hid, other_hid),
      DeviceSpecifier::XInput(other_xinput) => option_some_eq(&self.xinput, other_xinput),
      DeviceSpecifier::Websocket(other_websocket) => {
        option_some_eq(&self.websocket, other_websocket)
      }
      DeviceSpecifier::LovenseConnectService(other_lovense_service) => {
        option_some_eq(&self.lovense_connect_service, other_lovense_service)
      }
    }
  }
}

impl ProtocolDefinition {
  pub fn merge_user_definition(&mut self, other: ProtocolDefinition) {
    // Easy: Just extend vectors we already have
    if let Some(other_usb) = other.usb {
      if let Some(ref mut usb) = self.usb {
        usb.extend(other_usb);
      } else {
        self.usb = Some(other_usb);
      }
    }

    if let Some(other_serial) = other.serial {
      if let Some(ref mut serial) = self.serial {
        serial.extend(other_serial);
      } else {
        self.serial = Some(other_serial);
      }
    }

    if let Some(other_hid) = other.hid {
      if let Some(ref mut hid) = self.hid {
        hid.extend(other_hid);
      } else {
        self.hid = Some(other_hid);
      }
    }

    // Not so easy: Actually do complex merges for systems with more identifiers
    if let Some(other_btle) = other.btle {
      if let Some(ref mut btle) = self.btle {
        btle.merge(other_btle);
      } else {
        self.btle = Some(other_btle);
      }
    }

    if let Some(other_websocket) = other.websocket {
      if let Some(ref mut websocket) = self.websocket {
        websocket.merge(other_websocket);
      } else {
        self.websocket = Some(other_websocket);
      }
    }

    // Not possible: Don't even try to merge specific specifiers.
    if other.xinput.is_some() {
      error!("XInput specifier set for user configuration, ignoring.");
    }

    if other.lovense_connect_service.is_some() {
      error!("Lovense connect service specifier set for user configuration, ignoring.");
    }

    // If new defaults are set, overwrite.
    if other.defaults.is_some() {
      self.defaults = other.defaults;
    }

    // Treat configurations like paths; Extend using the new ones first, so we'll find them first,
    // but leave everything in. Post warning messages if anything repeats after this.
    if !other.configurations.is_empty() {
      self.configurations = other
        .configurations
        .iter()
        .chain(self.configurations.iter())
        .cloned()
        .collect();
    }
  }
}

#[derive(Clone, Debug)]
pub struct DeviceProtocolConfiguration {
  allow_raw_messages: bool,
  defaults: Option<ProtocolAttributes>,
  configurations: Vec<ProtocolAttributes>,
}

impl DeviceProtocolConfiguration {
  pub fn new(
    allow_raw_messages: bool,
    defaults: Option<ProtocolAttributes>,
    configurations: Vec<ProtocolAttributes>,
  ) -> Self {
    Self {
      allow_raw_messages,
      defaults,
      configurations,
    }
  }

  pub fn get_attributes(
    &self,
    identifier: &str,
    endpoints: &[Endpoint],
  ) -> Result<(HashMap<String, String>, DeviceMessageAttributesMap), ButtplugError> {
    let mut attributes = DeviceMessageAttributesMap::new();

    // If we find defaults, set those up first.
    if let Some(ref attrs) = self.defaults {
      if let Some(ref msg_attrs) = attrs.messages {
        attributes = msg_attrs.clone();
      }
    }

    // If we're allowing raw messages, tack those on beforehand also.
    if self.allow_raw_messages {
      let endpoint_attributes = DeviceMessageAttributes {
        endpoints: Some(endpoints.to_owned()),
        ..Default::default()
      };
      attributes.insert(
        ButtplugDeviceMessageType::RawReadCmd,
        endpoint_attributes.clone(),
      );
      attributes.insert(
        ButtplugDeviceMessageType::RawWriteCmd,
        endpoint_attributes.clone(),
      );
      attributes.insert(
        ButtplugDeviceMessageType::RawSubscribeCmd,
        endpoint_attributes.clone(),
      );
      attributes.insert(
        ButtplugDeviceMessageType::RawUnsubscribeCmd,
        endpoint_attributes,
      );
    }

    let device_attrs = if let Some(attrs) = self.configurations.iter().find(|attrs| {
      attrs
        .identifier
        .as_ref()
        .expect("Identifier required as part of JSON schema.")
        .contains(&identifier.to_owned())
    }) {
      attrs
    } else if let Some(attrs) = &self.defaults {
      // If we can't find an identifier but we have a default block, return that.
      attrs
    } else {
      // If we can't find anything, give up.
      return Err(
        ButtplugDeviceError::ProtocolAttributesNotFound(format!(
          "Cannot find identifier {} in protocol.",
          identifier
        ))
        .into(),
      );
    };

    if let Some(ref msg_attrs) = device_attrs.messages {
      attributes.extend(msg_attrs.clone());
    }

    // Everything needs to be able to stop.
    attributes
      .entry(ButtplugDeviceMessageType::StopDeviceCmd)
      .or_insert_with(DeviceMessageAttributes::default);

    Ok((
      device_attrs
        .name
        .as_ref()
        .expect("Name required as part of JSON schema")
        .clone(),
      attributes,
    ))
  }
}

pub struct DeviceConfigurationManager {
  allow_raw_messages: bool,
  protocol_definitions: Arc<DashMap<String, ProtocolDefinition>>,
  protocol_map: Arc<DashMap<String, TryCreateProtocolFunc>>,
}

impl Default for DeviceConfigurationManager {
  fn default() -> Self {
    // Unwrap allowed here because we assume our built in device config will
    // always work. System won't pass tests or possibly even build otherwise.
    Self::new(false)
  }
}

impl DeviceConfigurationManager {
  pub fn new(allow_raw_messages: bool) -> Self {
    Self {
      allow_raw_messages,
      protocol_definitions: Arc::new(DashMap::new()),
      protocol_map: Arc::new(get_default_protocol_map()),
    }
  }

  pub fn add_protocol_definition(
    &self,
    protocol_name: &str,
    protocol_definition: ProtocolDefinition,
  ) {
    self
      .protocol_definitions
      .insert(protocol_name.to_owned(), protocol_definition);
  }

  pub fn remove_protocol_definition(&self, protocol_name: &str) {
    self.protocol_definitions.remove(protocol_name);
  }

  pub fn add_protocol<T>(&self, protocol_name: &str)
  where
    T: ButtplugProtocol,
  {
    add_to_protocol_map::<T>(&self.protocol_map, protocol_name);
  }

  pub fn remove_protocol(&self, protocol_name: &str) {
    self.protocol_map.remove(protocol_name);
  }

  pub fn remove_all_protocols(&self) {
    self.protocol_map.clear();
  }

  pub fn has_protocol(&self, protocol_name: &str) -> bool {
    self.protocol_map.contains_key(protocol_name)
  }

  pub fn get_protocol_creator(&self, protocol_name: &str) -> Option<TryCreateProtocolFunc> {
    self
      .protocol_map
      .get(protocol_name)
      .map(|pair| *pair.value())
  }

  /// Provides read-only access to the internal protocol/identifier map. Mainly
  /// used for WebBluetooth filter construction, but could also be handy for
  /// listing capabilities in UI, etc.
  pub fn protocol_definitions(&self) -> Arc<DashMap<String, ProtocolDefinition>> {
    self.protocol_definitions.clone()
  }

  pub fn find_protocol_definitions(
    &self,
    specifier: &DeviceSpecifier,
  ) -> Option<(bool, String, ProtocolDefinition)> {
    debug!(
      "Looking for protocol that matches specifier: {:?}",
      specifier
    );
    for config in self.protocol_definitions.iter() {
      if config.value() == specifier {
        info!(
          "Found protocol {:?} for specifier {:?}.",
          config.key(),
          specifier
        );
        return Some((
          self.allow_raw_messages,
          config.key().clone(),
          config.value().clone(),
        ));
      }
    }
    debug!("No protocol found for specifier {:?}.", specifier);
    None
  }

  pub fn get_protocol_config(&self, name: &str) -> Option<DeviceProtocolConfiguration> {
    debug!("Looking for protocol {}", name);
    // TODO It feels like maybe there should be a cleaner way to do this,
    // but I'm not really sure what it is?
    if let Some(proto) = self.protocol_definitions.get(name) {
      info!("Found a protocol definition for {}", name);
      Some(DeviceProtocolConfiguration::new(
        self.allow_raw_messages,
        proto.defaults.clone(),
        proto.configurations.clone(),
      ))
    } else {
      debug!("No matching protocol definition found.");
      None
    }
  }
}

#[cfg(test)]
mod test {
  use super::{
    BluetoothLESpecifier,
    DeviceProtocolConfiguration,
    DeviceSpecifier,
    SerialSpecifier,
  };
  use crate::{
    core::messages::ButtplugDeviceMessageType,
    device::configuration_manager::ProtocolDefinition,
    util::device_configuration::create_test_dcm,
  };
  /*
    #[test]
    fn test_load_config() {
      let config = DeviceConfigurationManager::default();
      debug!("{:?}", config.config);
    }
  */
  #[test]
  fn test_config_equals() {
    let config = create_test_dcm(false);
    let launch = DeviceSpecifier::BluetoothLE(BluetoothLESpecifier::new_from_device("Launch", &[]));
    assert!(config.find_protocol_definitions(&launch).is_some());
  }

  #[test]
  fn test_config_wildcard_equals() {
    let config = create_test_dcm(false);
    let lovense =
      DeviceSpecifier::BluetoothLE(BluetoothLESpecifier::new_from_device("LVS-Whatever", &[]));
    assert!(config.find_protocol_definitions(&lovense).is_some());
  }

  #[test]
  fn test_specific_device_config_creation() {
    let config = create_test_dcm(false);
    let lovense =
      DeviceSpecifier::BluetoothLE(BluetoothLESpecifier::new_from_device("LVS-Whatever", &[]));
    let proto = config
      .find_protocol_definitions(&lovense)
      .expect("Test, assuming infallible");
    let proto_config =
      DeviceProtocolConfiguration::new(false, proto.2.defaults.clone(), proto.2.configurations);
    let (name_map, message_map) = proto_config
      .get_attributes("P", &vec![])
      .expect("Test, assuming infallible");
    // Make sure we got the right name
    assert_eq!(
      name_map.get("en-us").expect("Test, assuming infallible"),
      "Lovense Edge"
    );
    // Make sure we overwrote the default of 1
    assert_eq!(
      message_map
        .get(&ButtplugDeviceMessageType::VibrateCmd)
        .expect("Test, assuming infallible")
        .feature_count
        .expect("Test, assuming infallible"),
      2
    );
  }

  #[test]
  fn test_raw_device_config_creation() {
    let config = create_test_dcm(true);
    let lovense =
      DeviceSpecifier::BluetoothLE(BluetoothLESpecifier::new_from_device("LVS-Whatever", &[]));
    let proto = config
      .find_protocol_definitions(&lovense)
      .expect("Test, assuming infallible");
    let proto_config =
      DeviceProtocolConfiguration::new(true, proto.2.defaults.clone(), proto.2.configurations);
    let (name_map, message_map) = proto_config
      .get_attributes("P", &vec![])
      .expect("Test, assuming infallible");
    // Make sure we got the right name
    assert_eq!(
      name_map.get("en-us").expect("Test, assuming infallible"),
      "Lovense Edge"
    );
    // Make sure we overwrote the default of 1
    assert!(message_map.contains_key(&ButtplugDeviceMessageType::RawWriteCmd));
    assert!(message_map.contains_key(&ButtplugDeviceMessageType::RawReadCmd));
    assert!(message_map.contains_key(&ButtplugDeviceMessageType::RawSubscribeCmd));
    assert!(message_map.contains_key(&ButtplugDeviceMessageType::RawUnsubscribeCmd));
  }

  #[test]
  fn test_non_raw_device_config_creation() {
    let config = create_test_dcm(false);
    let lovense =
      DeviceSpecifier::BluetoothLE(BluetoothLESpecifier::new_from_device("LVS-Whatever", &[]));
    let proto = config
      .find_protocol_definitions(&lovense)
      .expect("Test, assuming infallible");
    let proto_config =
      DeviceProtocolConfiguration::new(false, proto.2.defaults.clone(), proto.2.configurations);
    let (name_map, message_map) = proto_config
      .get_attributes("P", &vec![])
      .expect("Test, assuming infallible");
    // Make sure we got the right name
    assert_eq!(
      name_map.get("en-us").expect("Test, assuming infallible"),
      "Lovense Edge"
    );
    // Make sure we overwrote the default of 1
    assert!(!message_map.contains_key(&ButtplugDeviceMessageType::RawWriteCmd));
    assert!(!message_map.contains_key(&ButtplugDeviceMessageType::RawReadCmd));
    assert!(!message_map.contains_key(&ButtplugDeviceMessageType::RawSubscribeCmd));
    assert!(!message_map.contains_key(&ButtplugDeviceMessageType::RawUnsubscribeCmd));
  }

  #[test]
  fn test_user_config_loading() {
    // Assume we have a nobra's entry in the device config.
    let mut config = create_test_dcm(false);
    assert!(config.protocol_definitions().contains_key("nobra"));
    assert!(config
      .protocol_definitions()
      .get("nobra")
      .expect("Test, assuming infallible")
      .serial
      .as_ref()
      .is_some());
    assert_eq!(
      config
        .protocol_definitions()
        .get("nobra")
        .expect("Test, assuming infallible")
        .serial
        .as_ref()
        .expect("Test, assuming infallible")
        .len(),
      1
    );

    // Now try overriding it, make sure we still only have 1.
    config = create_test_dcm(false);
    let mut nobra_def = ProtocolDefinition::default();
    let mut serial_specifier = SerialSpecifier::default();
    serial_specifier.port = "COM1".to_owned();
    nobra_def.serial = Some(vec![serial_specifier]);
    config.add_protocol_definition("nobra", nobra_def);
    assert!(config.protocol_definitions().contains_key("nobra"));
    assert!(config
      .protocol_definitions()
      .get("nobra")
      .expect("Test, assuming infallible")
      .serial
      .as_ref()
      .is_some());
    assert_eq!(
      config
        .protocol_definitions()
        .get("nobra")
        .expect("Test, assuming infallible")
        .serial
        .as_ref()
        .expect("Test, assuming infallible")
        .len(),
      1
    );
    assert!(config
      .protocol_definitions()
      .get("nobra")
      .expect("Test, assuming infallible")
      .serial
      .as_ref()
      .expect("Test, assuming infallible")
      .iter()
      .any(|x| x.port == "COM1"));
  }

  // TODO Test invalid config load (not json)
  // TODO Test invalid user config load (not json)
  // TODO Test device config with repeated ble service
  // TODO Test device config with repeated ble characteristic
  // TODO Test user config with invalid protocol
  // TODO Test user config with invalid bus type
  // TODO Test user config with conflicting BLE name
}
