#[cfg(not(target_arch = "wasm32"))]
mod pipe;
#[cfg(feature = "websockets")]
mod websocket;
use crate::connector::{
  ButtplugConnectorError,
  ButtplugConnectorResultFuture,
  ButtplugSerializedMessage,
};
use futures::future::BoxFuture;
#[cfg(not(target_arch = "wasm32"))]
pub use pipe::{
  pipe_client::{ButtplugPipeClientTransport, ButtplugPipeClientTransportBuilder},
  pipe_server::{ButtplugPipeServerTransport, ButtplugPipeServerTransportBuilder},
};
use thiserror::Error;
use tokio::sync::mpsc::{Receiver, Sender};
#[cfg(feature = "websockets")]
pub use websocket::{
  ButtplugWebsocketClientTransport,
  ButtplugWebsocketServerTransport,
  ButtplugWebsocketServerTransportBuilder,
  TungsteniteError,
};

/// Messages we can receive from a connector.
#[derive(Clone, Debug, Display)]
pub enum ButtplugTransportIncomingMessage {
  /// Send when connection is established.
  Connected,
  /// Serialized version of message we received from remote server.
  Message(ButtplugSerializedMessage),
  // TODO Implement binary message at some point.
  /// Error received from remote server.
  Error(String),
  /// Connector (or remote server) itself closed the connection.
  Close(String),
}

pub trait ButtplugConnectorTransport: Send + Sync {
  fn connect(
    &self,
    outgoing_receiver: Receiver<ButtplugSerializedMessage>,
    incoming_sender: Sender<ButtplugTransportIncomingMessage>,
  ) -> BoxFuture<'static, Result<(), ButtplugConnectorError>>;
  fn disconnect(self) -> ButtplugConnectorResultFuture;
}

#[derive(Error, Debug)]
pub enum ButtplugConnectorTransportSpecificError {
  #[cfg(feature = "websockets")]
  #[error("Tungstenite specific error: {0}")]
  TungsteniteError(#[from] TungsteniteError),
  #[error("Network error: {0}")]
  GenericNetworkError(String),
}
