use std::borrow::Borrow;
use std::cell::{Ref, RefCell};
use std::collections::HashMap;
use std::fs;
use std::io::Write;
use std::path::Path;
use std::rc::Rc;
use skia_safe::{BlendMode, Canvas, Color, Color4f, Data, EncodedImageFormat, FontMgr, FontStyle, Paint, Rect, Surface, Typeface};
use skia_safe::textlayout::{FontCollection, Paragraph, ParagraphBuilder, ParagraphStyle, TextStyle, TypefaceFontProvider};
use stretch::geometry::{Point, Size};
use stretch::node::Node;
use stretch::{Error, Stretch};
use stretch::number::{Number, OrElse};

use stretch::style::{AlignItems, AlignSelf, Dimension, Display, JustifyContent, Style};
use crate::models::{ChildUpdate, ComputedStyle, MyNode, MyNodeChange, MyNodeId, SpecifiedStyle};
use crate::parsers::css::Stylesheet;
use css_color_parser::Color as CssColor;
use skia_safe::font_style::{Slant, Weight, Width};
use crate::models::Property::FontWeight;


#[derive(Debug, Clone)]
pub struct ElementRenderObject {
    pub class: String,
    pub style: ComputedStyle,
}

#[derive(Debug, Clone)]
pub struct TextRenderObject {
    pub text: String,
    pub paragraph: Option<Rc<Paragraph>>,
    pub style: ComputedStyle,
}

impl TextRenderObject {
    pub fn layout_paragraph(&mut self, fonts: FontCollection, width: f32) {
        let paragraph_style = ParagraphStyle::new();
        let mut builder = ParagraphBuilder::new(&paragraph_style, fonts);
        let font_style = FontStyle::new(
            Weight::LIGHT,
            Width::NORMAL,
            Slant::Upright
        );
        let mut text_style = TextStyle::new();
        text_style.set_font_families(&["Fira Sans"]);
        text_style.set_font_size(20.0);
        text_style.set_foreground_color(Paint::default());
        text_style.set_font_style(font_style);
        builder.push_style(&text_style);
        builder.add_text(&self.text);
        let mut paragraph = builder.build();
        paragraph.layout(width);
        self.paragraph = Some(Rc::new(paragraph));
    }
}

pub type TextRenderObjectRef = Rc<RefCell<TextRenderObject>>;

#[derive(Debug, Clone)]
pub enum RenderObject {
    Element(ElementRenderObject),
    Text(TextRenderObjectRef),
}

pub struct Document {
    tree: Stretch,
    root: Node,
    fonts: FontCollection,
    objects: HashMap<Node, RenderObject>,
    stylesheet: Stylesheet
}



impl Document {
    // https://developer.chrome.com/blog/inside-browser-part3/

    pub fn new(width: f32, height: f32, stylesheet: Stylesheet) -> Self {
        let mut dom = Stretch::new();
        let root = Style {
            display: Display::Flex,
            size: Size {
                width: Dimension::Points(width),
                height: Dimension::Points(height),
            },
            ..Style::default()
        };
        let root = dom.new_node(root, vec![]).unwrap();
        let mut objects = HashMap::new();
        objects.insert(root, RenderObject::Element( ElementRenderObject {
            class: ":root".to_string(),
            style: Default::default()
        }));

        // fonts
        let mut font_collection = FontCollection::new();
        let mut manager = TypefaceFontProvider::new();
        font_collection.set_default_font_manager(FontMgr::new(), Some("Ubuntu"));

        let buffer = include_bytes!("../assets/fira/FiraSans-Light.ttf");
        let data = Data::new_copy(buffer);
        let mut typeface = Typeface::from_data(data, 0).unwrap();
        println!("Fira Light {:?}, {}, {}, {}",
            typeface.font_style(),
            typeface.family_name(),
            typeface.is_bold(),
            typeface.is_italic()
            );

        let alias: Option<&str> = None;
        manager.register_typeface(typeface, alias);

        let buffer = include_bytes!("../assets/fira/FiraSans-Regular.ttf");
        let data = Data::new_copy(buffer);
        let mut typeface = Typeface::from_data(data, 0).unwrap();
        println!("Fira Regular {:?}, {}, {}, {}",
                 typeface.font_style(),
                 typeface.family_name(),
                 typeface.is_bold(),
                 typeface.is_italic(),
        );
        let alias: Option<&str> = None;
        manager.register_typeface(typeface, alias);

        let buffer = include_bytes!("../assets/fira/FiraSans-Bold.ttf");
        let data = Data::new_copy(buffer);
        let mut typeface = Typeface::from_data(data, 0).unwrap();
        println!("Fira Bold {:?}, {}, {}, {}",
                 typeface.font_style(),
                 typeface.family_name(),
                 typeface.is_bold(),
                 typeface.is_italic()
        );
        let alias: Option<&str> = None;
        manager.register_typeface(typeface, alias);

        font_collection.set_asset_font_manager(Some(manager.into()));
        //

        Self {
            tree: dom,
            root,
            fonts: font_collection,
            objects,
            stylesheet
        }
    }

    pub fn root(&self) -> Node {
        self.root
    }

    pub fn render(&mut self, node: MyNode) -> Node {
        match node {
            MyNode::Element(element) => {
                let style = Style {
                    display: Display::Flex,
                    flex_grow: 1.0,
                    margin: stretch::geometry::Rect {
                        start: Dimension::Points(20.0),
                        end: Dimension::Points(20.0),
                        top: Dimension::Points(20.0),
                        bottom: Dimension::Points(20.0)
                    },
                    align_items: AlignItems::Stretch,
                    size: Size {
                        width: Dimension::Undefined,
                        height: Dimension::Undefined
                    },
                    ..Style::default()
                };
                let children: Vec<Node> = element.children.into_iter().map(|node| self.render(node)).collect();
                let node = self.tree.new_node(style, children.clone()).unwrap();
                self.objects.insert(node, RenderObject::Element(ElementRenderObject {
                    class: element.class.to_string(),
                    style: Default::default(),
                }));
                node
            }
            MyNode::Text(text) => {
                let style = Style {
                    display: Display::Flex,
                    flex_grow: 1.0,
                    align_self: AlignSelf::Stretch,
                    size: Size {
                        width: Dimension::Undefined,
                        height: Dimension::Undefined
                    },
                    ..Style::default()
                };
                let object = Rc::new(RefCell::new(TextRenderObject {
                    text: text.value.to_string(),
                    paragraph: None,
                    style: ComputedStyle::default()
                }));
                let measure_object = object.clone();
                let measure_fonts = self.fonts.clone();
                let measure = Box::new(move |constraint: Size<Number>| {
                    let metrics = match constraint.width {
                        Number::Defined(width) => {
                            let mut object = RefCell::borrow_mut(&measure_object);
                            object.layout_paragraph(measure_fonts.clone(), width);
                            let paragraph = &object.paragraph.as_ref().unwrap();
                            println!("h: {}, constraint: {:?}", paragraph.height(), constraint);
                            Size {
                                width,
                                height: paragraph.height(),
                            }
                        }
                        Number::Undefined => {
                            Size {
                                width: constraint.width.or_else(0.0),
                                height: constraint.height.or_else(0.0),
                            }
                        }
                    };
                    Ok(metrics)
                });

                let node = self.tree.new_leaf(style, measure).unwrap();
                self.objects.insert(node, RenderObject::Text(object));
                node
            }
        }
    }

    pub fn paint_to(&self, width: i32, height: i32, name: &str) {
        let mut surface = Surface::new_raster_n32_premul((width, height)).unwrap();
        let canvas = surface.canvas();
        canvas.scale((1.0, 1.0));
        self.paint(Point {x: 0.0, y: 0.0}, self.root, canvas);
        let image = surface.image_snapshot();
        let data = image.encode_to_data(EncodedImageFormat::PNG).unwrap();
        write_file(data.as_bytes(), Path::new("./output"), name, "png");
    }

    pub fn paint(&self, offset: Point<f32>, node: Node, canvas: &mut Canvas) {
        let layout = self.tree.layout(node).unwrap();
        let location = Point {
            x: layout.location.x + offset.x,
            y: layout.location.y + offset.y
        };
        let rect = Rect::new(
            location.x,
            location.y,
            location.x + layout.size.width,
            location.y + layout.size.height
        );
        println!("{:?}, {:?}, {:?} loc:{:?}", node, rect, layout, location);
        let object = self.objects.get(&node).unwrap();

        match object {
            RenderObject::Element(element) => {
                let style = &element.style;
                // is not transparent
                if style.background_color.3 > 0.0 {
                    let mut paint = Paint::default();
                    paint.set_color(to_color(style.background_color));
                    canvas.draw_rect(rect, &paint);
                }
                for child in self.tree.children(node).unwrap() {
                    self.paint(location, child, canvas);
                }
            }
            RenderObject::Text(text) => {
                let object = RefCell::borrow(&text);
                let paragraph = object.paragraph.as_ref().unwrap();
                paragraph.paint(canvas, skia_safe::Point::new(location.x, location.y));
            }
        }
    }

    pub fn apply(&mut self, current: Node, change: MyNodeChange) -> Node {
        match change {
            MyNodeChange::Update(change) => {
                let mut offset = 0;
                for (index, change) in change.children.into_iter().enumerate() {
                    match change {
                        ChildUpdate::Insert(insert) => {
                            let mut child_nodes: Vec<Node> = self.tree.children(current).unwrap();
                            match child_nodes.get(index + offset) {
                                Some(_sibling) => {
                                    let new = self.render(insert.node);
                                    child_nodes.insert(index + offset, new);
                                    self.tree.set_children(current, child_nodes);
                                }
                                None => {
                                    let new = self.render(insert.node);
                                    self.tree.add_child(current, new);
                                }
                            }
                        }
                        ChildUpdate::Update(change) => {
                            let child = self.tree.children(current).unwrap()[index + offset];
                            self.apply(child, MyNodeChange::Update(change));
                        }
                        ChildUpdate::Replace(replace) => {
                            let new = self.render(replace.node);
                            self.tree.replace_child_at_index(current, index + offset, new).unwrap();
                        }
                        ChildUpdate::Remove => {
                            self.tree.remove_child_at_index(current, index + offset).unwrap();
                            offset -= 1;
                        }
                        ChildUpdate::Skip => {}
                    }
                }
                current
            }
            MyNodeChange::Replace(_) => {
                unimplemented!()
            }
            MyNodeChange::Skip => current
        }
    }

    pub fn calculate_styles(&mut self) {
        let cascading = SpecifiedStyle::new();
        self.calc_style(self.root, cascading)
    }

    pub fn calc_style(&mut self, node: Node, cascading: SpecifiedStyle) {
        let object = self.objects.get_mut(&node).unwrap();
        match object {
            RenderObject::Element(element) => {
                let specified_style = self.stylesheet.match_style(&element.class);
                element.style = specified_style.compute(&cascading, &cascading);
                for child in self.tree.children(node).unwrap() {
                    let cascading = cascading.merge(&specified_style);
                    self.calc_style(child, cascading);
                }
            }
            RenderObject::Text(text) => {
                text.borrow_mut().style = cascading.compute(&cascading, &cascading);
            }
        }
    }

    pub fn layout(&mut self) -> Result<(), Error> {
        self.tree.compute_layout(self.root, Size::undefined())
    }
}

fn write_file(bytes: &[u8], path: &Path, name: &str, ext: &str) {
    fs::create_dir_all(&path).expect("failed to create directory");
    let mut file_path = path.join(name);
    file_path.set_extension(ext);
    let mut file = fs::File::create(file_path).expect("failed to create file");
    file.write_all(bytes).expect("failed to write to file");
}



pub fn to_color(tuple: (f32, f32, f32, f32)) -> Color {
    fn c(f: f32) -> u8 {
        (f.max(0.0).min(1.0) * 255.0) as u8
    }
    let a = c(tuple.3);
    let r = c(tuple.0);
    let g = c(tuple.1);
    let b = c(tuple.2);
    Color::from_argb(a, r, g, b)
}