use std::fs::File;
use std::io::Write;
use pest::{error, Parser};
use pest::iterators::Pair;
use pest::RuleType;
use serde_json::{Map, Value};
use serde::de::DeserializeOwned;
use serde::{Serialize, Deserialize};
use crate::models::{Property, SpecifiedStyle};


mod gen {
    /// https://drafts.csswg.org/css-syntax/
    #[derive(Parser)]
    #[grammar = "./grammars/css.pest"]
    pub struct CssParser;
}

#[derive(Debug)]
pub struct Error {}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Selector {
    pub ident: String
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Rule {
    pub Selector: Selector,
    pub properties: Vec<Property>
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Stylesheet {
    pub rules: Vec<Rule>
}

impl Stylesheet {
    pub fn match_style(&self, class: &String) -> SpecifiedStyle {
        let mut style = SpecifiedStyle::new();
        for rule in &self.rules {
            if &rule.Selector.ident == class {
                style.properties.extend(rule.properties.clone())
            }
        }
        style
    }
}

pub fn parse_css(source: &str) -> Result<Stylesheet, Error> {
    let declarations = gen::CssParser::parse(gen::Rule::Stylesheet, source)?
        .next()
        .ok_or(Error {})?;

    let arrays = vec!["rules".to_string(), "properties".to_string()];
    let converter = JsonConverter::new(arrays);

    let value = converter.create_value(declarations);
    let file = File::create("./output/css.json").unwrap();
    serde_json::to_writer_pretty(file, &value).unwrap();

    let stylesheet = serde_json::from_value(value).unwrap();

    Ok(stylesheet)
}

impl From<error::Error<gen::Rule>> for Error {
    fn from(error: error::Error<gen::Rule>) -> Self {
        Error {}
    }
}

pub struct JsonConverter {
    pub arrays: Vec<String>,
}

impl JsonConverter {
    pub fn new(arrays: Vec<String>) -> Self {
        JsonConverter { arrays }
    }

    pub fn convert<T, R>(&self, pair: Pair<'_, R>) -> Result<T, serde_json::Error>
        where
            T: DeserializeOwned,
            R: RuleType,
    {
        let value = self.create_value(pair);
        serde_json::from_value(value)
    }

    pub fn create_value<R>(&self, pair: Pair<'_, R>) -> Value
        where
            R: RuleType,
    {
        let rule = format!("{:?}", pair.as_rule());
        let data = pair.as_str();
        let inner = pair.into_inner();
        if inner.peek().is_none() {
            if self.arrays.contains(&rule) {
                Value::Array(vec![])
            } else {
                Value::String(data.into())
            }
        } else {
            if self.arrays.contains(&rule) {
                let values = inner.map(|pair| self.create_value(pair)).collect();
                Value::Array(values)
            } else {
                let map = inner.map(|pair| {
                    let key = format!("{:?}", pair.as_rule());
                    let value = self.create_value(pair);
                    (key, value)
                });
                Value::Object(Map::from_iter(map))
            }
        }
    }
}
