use std::cell::RefCell;
use serde::{Serialize, Deserialize};

use std::collections::HashMap;
use std::rc::Rc;
use std::str::FromStr;
use skia_safe::{Font, Paint, RefHandle};
use skia_safe::textlayout::{FontCollection, ParagraphBuilder, ParagraphStyle, TextStyle};
use stretch::geometry::Size;
use stretch::node::{MeasureFunc, Node};
use stretch::number::{Number, OrElse};
use css_color_parser::Color as CssColor;
use skia_safe::font_style::Weight;
use crate::render::TextRenderObject;


pub struct MyNodeId(Node);

pub enum MyNode {
    Element(MyElement),
    Text(MyText),
}

pub struct MyElement {
    pub class: String,
    pub children: Vec<MyNode>,
}

impl MyElement {
    pub fn create<T: ToString>(class: T) -> MyNode {
        MyNode::Element(Self {
            class: class.to_string(),
            children: vec![],
        })
    }

    pub fn with_children<T: ToString>(class: T, children: Vec<MyNode>) -> MyNode {
        MyNode::Element(Self {
            class: class.to_string(),
            children,
        })
    }

    pub fn with_text<T: ToString>(class: T, text: T) -> MyNode {
        Self::with_children(class, vec![MyText::create(text)])
    }
}

pub struct MyText {
    pub value: String,
}

impl MyText {
    pub fn create<T: ToString>(value: T) -> MyNode {
        MyNode::Text(Self {
            value: value.to_string(),
        })
    }
}

#[derive(Default)]
pub struct MyComputedStyle {
    pub flex_grow: f32,
    pub flex_shrink: f32,
}

pub struct InsertInstruction {
    pub node: MyNode,
}

pub struct UpdateInstruction {
    pub children: Vec<ChildUpdate>,
}

pub struct ReplaceInstruction {
    pub node: MyNode,
}

pub enum ChildUpdate {
    Insert(InsertInstruction),
    Update(UpdateInstruction),
    Replace(ReplaceInstruction),
    Remove,
    Skip,
}

pub enum MyNodeChange {
    Update(UpdateInstruction),
    Replace(ReplaceInstruction),
    Skip,
}

#[derive(Debug, Copy, Clone, PartialEq, Hash)]
pub enum MyKey {
    Color,
    Background,
}

pub struct MyStyleSheet {
    parent: Option<Rc<MyStyleSheet>>,

    color: Option<f32>,
    background: Option<f32>,
}

impl MyStyleSheet {
    pub fn color(&self) -> f32 {
        match self.color {
            Some(value) => value,
            None => match &self.parent {
                Some(parent) => parent.color(),
                None => 0.0,
            }
        }
    }
}


type Image = String;

#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum Color {
    Color(String)
}

impl Default for Color {
    fn default() -> Self {
        Self::Color(String::default())
    }
}

pub fn compute_color(value: &Color) -> (f32, f32, f32, f32) {
    (0.0, 0.0, 0.0, 0.0)
}

pub fn compute_background_size(value: &SpecBackgroundSize) -> ComputedBackgroundSize {
    ComputedBackgroundSize::Auto
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum SpecBackgroundSize {
    Length(String),
    Percentage(String),
    Auto
}


///
/// see:
/// - https://www.w3.org/TR/css-2021/#properties
#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum Property {
    Background(String),
    BackgroundAttachment(String),
    BackgroundBlendMode(String),
    BackgroundClip(String),
    BackgroundColor(Color),
    BackgroundImage(Vec<Image>),
    BackgroundOrigin(String),
    BackgroundPosition(String),
    BackgroundRepeat(String),
    BackgroundSize(Vec<SpecBackgroundSize>),
    // TODO:
    Border(String),
    BorderColor(Vec<Color>),
    BorderTopColor(Color),
    BorderRightColor(Color),
    BorderBottomColor(Color),
    BorderLeftColor(Color),
    // TODO:
    Font,
    FontFamily,
    FontFeatureSettings,
    FontKerning,
    FontSize,
    FontSizeAdjust,
    FontStretch,
    FontStyle,
    FontSynthesis,
    FontVariant,
    FontVariantCaps,
    FontVariantEastAsian,
    FontVariantLigatures,
    FontVariantNumeric,
    FontVariantPosition,
    FontWeight(String)
}


#[derive(Serialize, Deserialize)]
pub struct SpecifiedStyle {
    pub properties: Vec<Property>,
}

impl SpecifiedStyle {
    pub fn new() -> Self {
        Self {
            properties: vec![]
        }
    }

    pub fn merge(&self, other: &SpecifiedStyle) -> Self {
        let mut properties = self.properties.clone();
        properties.extend(other.properties.clone());
        Self {
            properties
        }
    }

    pub fn compute(&self, cascade: &SpecifiedStyle, parent: &SpecifiedStyle) -> ComputedStyle {
        let mut style = ComputedStyle::default();

        for property in &self.properties {
            match property {
                Property::Background(_) => {}
                Property::BackgroundAttachment(_) => {}
                Property::BackgroundBlendMode(_) => {}
                Property::BackgroundClip(_) => {}
                Property::BackgroundColor(repr) => {
                    let color = if let Color::Color(repr) = repr {
                        CssColor::from_str(repr).unwrap()
                    } else {
                        CssColor::from_str("transparent").unwrap()
                    };
                    style.background_color = (color.r as f32 / 255.0, color.g as f32 / 255.0, color.b as f32 / 255.0, color.a);

                }
                Property::BackgroundImage(images) => {
                    style.background_images = images.clone();
                }
                Property::BackgroundOrigin(_) => {}
                Property::BackgroundPosition(_) => {}
                Property::BackgroundRepeat(_) => {}
                Property::BackgroundSize(sizes) => {
                    let default = SpecBackgroundSize::Auto;
                    let (width, height) = match sizes.len() {
                        2 => (&sizes[0], &sizes[1]),
                        1 => (&sizes[0], &default),
                        _ => {
                            println!("BackgroundSize incorrect! failback to default");
                            (&default, &default)
                        }
                    };
                    style.background_size_width = compute_background_size(width);
                    style.background_size_height = compute_background_size(height);
                }
                // TODO:
                Property::Border(_) => {}
                Property::BorderColor(colors) => {
                    let default = Color::default();
                    let (top, bottom, left, right) = match colors.len() {
                        4 => (&colors[0], &colors[0], &colors[0], &colors[0]),
                        3 => (&colors[0], &colors[0], &colors[0], &colors[0]),
                        2 => (&colors[0], &colors[0], &colors[0], &colors[0]),
                        1 => (&colors[0], &colors[0], &colors[0], &colors[0]),
                        _ => {
                            println!("BorderColor incorrect! failback to default");
                            (&default, &default, &default, &default)
                        }
                    };
                    style.border_top_color = compute_color(top);
                    style.border_bottom_color = compute_color(bottom);
                    style.border_left_color = compute_color(left);
                    style.border_right_color = compute_color(right);
                }
                Property::BorderTopColor(color) => {
                    style.border_top_color = compute_color(&color);
                }
                Property::BorderRightColor(color) => {
                    style.border_right_color = compute_color(&color);
                }
                Property::BorderBottomColor(color) => {
                    style.border_bottom_color = compute_color(&color);
                }
                Property::BorderLeftColor(color) => {
                    style.border_left_color = compute_color(&color);
                }
                Property::Font => {}
                Property::FontFamily => {}
                Property::FontFeatureSettings => {}
                Property::FontKerning => {}
                Property::FontSize => {}
                Property::FontSizeAdjust => {}
                Property::FontStretch => {}
                Property::FontStyle => {}
                Property::FontSynthesis => {}
                Property::FontVariant => {}
                Property::FontVariantCaps => {}
                Property::FontVariantEastAsian => {}
                Property::FontVariantLigatures => {}
                Property::FontVariantNumeric => {}
                Property::FontVariantPosition => {}
                Property::FontWeight(value) => {
                    style.font_weight = match &value[..] {
                        "100" => Weight::THIN,
                        "200" => Weight::EXTRA_LIGHT,
                        "300" => Weight::LIGHT,
                        "400" => Weight::NORMAL,
                        "500" => Weight::MEDIUM,
                        "600" => Weight::SEMI_BOLD,
                        "700" => Weight::BOLD,
                        "800" => Weight::EXTRA_BOLD,
                        "900" => Weight::BLACK,
                        "normal" => Weight::NORMAL,
                        "bold" => Weight::BOLD,
                        "bolder" => {
                            warn!("font weight 'bolder' not implemented yet, use bold");
                            Weight::BOLD
                        },
                        "lighter" => {
                            warn!("font weight 'bolder' not implemented yet, use light");
                            Weight::LIGHT
                        },
                        value => {
                            warn!("unexpected font weight value '{}', use normal", value);
                            Weight::NORMAL
                        },
                    }
                }
            }
        }

        style
    }
}

#[derive(Debug, Clone)]
pub enum ComputedBackgroundSize {
    Length(f32),
    Percentage(f32),
    Auto
}

impl Default for ComputedBackgroundSize {
    fn default() -> Self {
        Self::Auto
    }
}

#[derive(Debug, Clone)]
pub struct ComputedStyle {
    pub background_color: (f32, f32, f32, f32),
    pub background_images: Vec<Image>,
    pub background_size_width: ComputedBackgroundSize,
    pub background_size_height: ComputedBackgroundSize,
    pub border_top_color: (f32, f32, f32, f32),
    pub border_bottom_color: (f32, f32, f32, f32),
    pub border_left_color: (f32, f32, f32, f32),
    pub border_right_color: (f32, f32, f32, f32),
    pub font: usize,
    pub font_family: usize,
    pub font_feature_settings: usize,
    pub font_kerning: usize,
    pub font_size: usize,
    pub font_size_adjust: usize,
    pub font_stretch: usize,
    pub font_style: usize,
    pub font_synthesis: usize,
    pub font_variant: usize,
    pub font_variant_caps: usize,
    pub font_variant_east_asian: usize,
    pub font_variant_ligatures: usize,
    pub font_variant_numeric: usize,
    pub font_variant_position: usize,
    pub font_weight: Weight,
}

impl Default for ComputedStyle {
    fn default() -> Self {
        Self {
            background_color: (0.0, 0.0, 0.0, 0.0),
            background_images: vec![],
            background_size_width: Default::default(),
            background_size_height: Default::default(),
            border_top_color: (0.0, 0.0, 0.0, 0.0),
            border_bottom_color: (0.0, 0.0, 0.0, 0.0),
            border_left_color: (0.0, 0.0, 0.0, 0.0),
            border_right_color: (0.0, 0.0, 0.0, 0.0),
            font: 0,
            font_family: 0,
            font_feature_settings: 0,
            font_kerning: 0,
            font_size: 0,
            font_size_adjust: 0,
            font_stretch: 0,
            font_style: 0,
            font_synthesis: 0,
            font_variant: 0,
            font_variant_caps: 0,
            font_variant_east_asian: 0,
            font_variant_ligatures: 0,
            font_variant_numeric: 0,
            font_variant_position: 0,
            font_weight: Weight::NORMAL
        }
    }
}

