use num_bigint::BigUint;
use std::io::Write;

use std::{
	env,
	fs::File,
	io::{self, BufRead},
	path::Path,
};

use crate::poseidon::{PoseidonBuilder, PoseidonSbox, Poseidon_hash_2};
use curve25519_dalek::scalar::Scalar;

// NOTE: Code used to generate ZERO_TREE
pub fn generate_zero_trees() {
	let curr_dir = env::current_dir().unwrap();
	let width = 6;
	let base = format!("{}/src/crypto_constants/smt", curr_dir.display());
	write_zero_tree(
		width,
		&PoseidonSbox::Exponentiation3,
		format!("{}/x3_{}.rs", base, width),
	);
	write_zero_tree(
		width,
		&PoseidonSbox::Exponentiation5,
		format!("{}/x5_{}.rs", base, width),
	);
	write_zero_tree(
		width,
		&PoseidonSbox::Exponentiation17,
		format!("{}/x17_{}.rs", base, width),
	);
	write_zero_tree(
		width,
		&PoseidonSbox::Inverse,
		format!("{}/inverse_{}.rs", base, width),
	);
}

pub fn write_zero_tree(width: usize, sbox: &PoseidonSbox, path: String) {
	let hasher = PoseidonBuilder::new(width).sbox(*sbox).build();

	let mut curr = Scalar::zero();
	let mut tree = vec![curr.to_bytes()];
	for _ in 0..256 {
		curr = Poseidon_hash_2(curr, curr, &hasher);
		tree.push(curr.to_bytes());
	}

	let mut data = "pub const ZERO_TREE: [[u8; 32]; 257] =".to_string();
	data.push_str(&format!("{:?}", tree));
	data.push_str(";");

	println!("{:?}", path);
	let mut file = File::create(path).unwrap();
	file.write_all(data.as_bytes()).unwrap();
}

pub fn print_constants() {
	for i in 2..10 {
		let p = format!("./data/poseidon/round_constants_{}.txt", i);
		// some formatting by hand necessary
		let _o = format!("./src/poseidon/round_constants_{}.rs", i);
		if let Ok(lines) = read_lines(p) {
			let mut v = Vec::new();
			// Consumes the iterator, returns an (Optional) String
			for line in lines {
				if let Ok(num) = line {
					let val: BigUint =
						std::str::FromStr::from_str(&num).unwrap();
					let mut s = hex::encode(val.to_bytes_le());
					s = "0x".to_owned() + &s;
					v.push(s);
				}
			}

			// we've already written the files! some manually editting was added
			// let _ = write_file(o, v);
		}

		let temp: Vec<Vec<String>> = read_mds(i)
			.iter()
			.map(|v| {
				v.iter()
					.map(|e| "0x".to_owned() + &hex::encode(e.to_bytes_le()))
					.collect()
			})
			.collect();
		println!("----------\n{:?}: {:?}\n-----------", i, temp);
	}
}

// The output is wrapped in a Result to allow matching on errors
// Returns an Iterator to the Reader of the lines of the file.
pub fn read_lines<P>(filename: P) -> io::Result<io::Lines<io::BufReader<File>>>
where
	P: AsRef<Path>,
{
	let file = File::open(filename)?;
	Ok(io::BufReader::new(file).lines())
}

pub fn write_file<
	T: std::fmt::Display,
	P: std::convert::AsRef<std::path::Path>,
>(
	filename: P,
	data: Vec<T>,
) -> std::io::Result<()> {
	let mut file = File::create(filename).expect("Unable to create file");
	for i in &data {
		let _ = write!(file, "{}\n", i);
	}
	Ok(())
}

pub fn read_mds(i: usize) -> Vec<Vec<num_bigint::BigUint>> {
	if i == 2 {
		return [
			[
				std::str::FromStr::from_str(
					"2910766817845651019878574839501801340070030115151021261302834310722729507541",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"19727366863391167538122140361473584127147630672623100827934084310230022599144",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"5776684794125549462448597414050232243778680302179439492664047328281728356345",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"8348174920934122550483593999453880006756108121341067172388445916328941978568",
				)
				.unwrap(),
			]
			.to_vec(),
		]
		.to_vec();
	} else if i == 3 {
		return [
			[
				std::str::FromStr::from_str(
					"7511745149465107256748700652201246547602992235352608707588321460060273774987",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10370080108974718697676803824769673834027675643658433702224577712625900127200",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"19705173408229649878903981084052839426532978878058043055305024233888854471533",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"18732019378264290557468133440468564866454307626475683536618613112504878618481",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20870176810702568768751421378473869562658540583882454726129544628203806653987",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7266061498423634438633389053804536045105766754026813321943009179476902321146",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"9131299761947733513298312097611845208338517739621853568979632113419485819303",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10595341252162738537912664445405114076324478519622938027420701542910180337937",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11597556804922396090267472882856054602429588299176362916247939723151043581408",
				)
				.unwrap(),
			]
			.to_vec(),
		]
		.to_vec();
	} else if i == 4 {
		return [
			[
				std::str::FromStr::from_str(
					"16023668707004248971294664614290028914393192768609916554276071736843535714477",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17849615858846139011678879517964683507928512741474025695659909954675835121177",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1013663139540921998616312712475594638459213772728467613870351821911056489570",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"13211800058103802189838759488224684841774731021206389709687693993627918500545",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"19204974983793400699898444372535256207646557857575315905278218870961389967884",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3722304780857845144568029505892077496425786544014166938942516810831732569870",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11920634922168932145084219049241528148129057802067880076377897257847125830511",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"6085682566123812000257211683010755099394491689511511633947011263229442977967",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"14672613178263529785795301930884172260797190868602674472542654261498546023746",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20850178060552184587113773087797340350525370429749200838012809627359404457643",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7082289538076771741936674361200789891432311337766695368327626572220036527624",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1787876543469562003404632310460227730887431311758627706450615128255538398187",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"21407770160218607278833379114951608489910182969042472165261557405353704846967",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16058955581309173858487265533260133430557379878452348481750737813742488209262",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"593311177550138061601452020934455734040559402531605836278498327468203888086",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"341662423637860635938968460722645910313598807845686354625820505885069260074",
				)
				.unwrap(),
			]
			.to_vec(),
		]
		.to_vec();
	} else if i == 5 {
		return [
			[
				std::str::FromStr::from_str(
					"16789463359527776692258765063233607350971630674230623383979223533600140787105",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17179611066821656668705197789232102741366879862607190942874777813024566441829",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18653277315487164762584377009009109585010878033606596417396490909822722930739",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7373070639853668650581790286343199505413793790160702463077019294817051722180",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"4823864393442908763804841692709014014130031798360007432734996408628916373879",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"19196309854577132760746782449135315310664418272926255500908899397538686486585",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18123132816088485879885148351452823314623055244145916622592591084094232513914",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18436594886553181913092702411547018228276047601279727265790147051821171174455",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15167500404313194506503404655898040457721633218143681920692711693000769735187",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"9437986152015460505719924283993842205604222075968464846270136901243896809793",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"21445376105821232747280055223032050399373725161014449207033808524504027971613",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"49684738714301073369749035791061182456037935161360748355432247732088942674",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"9826409059947591908303145327284336313371973037536805760095514429930589897515",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"8494798325496773219358794086647759478982958403252584257436898618394561204124",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21251937175072447337747316555423152807036003235223125066270735279039060889959",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"5539100337780919206842837176908516952801756637410959104376645017856664270896",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"6297628909516159190915174165284309160976659474973668336571577778869958189934",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12792263637464508665199868777503118105486490400267592501708855807938962470650",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17254685306085558791725544672172906900581495686070720065168939143671412445514",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3590396502942934679818900672232030233017710909687947858184099000783280809247",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"19055249881366445073616526879263250763682650596233071589085239500077496415637",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7367697936402141224946246030743627391716576575953707640061577218995381577033",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1322791522030759131093883057746095061798181102708855007233180025036972924046",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20456741074925985565499300081580917471340328842103779922028754640077047587707",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"9059147312071680695674575245237100802111605600478121517359780850134328696420",
				)
				.unwrap(),
			]
			.to_vec(),
		]
		.to_vec();
	} else if i == 6 {
		return [
			[
				std::str::FromStr::from_str(
					"8266021233794274332054729525918686051968756165685671155584565440479247355160",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7947823415909040438587565055355894256799314737783432792935458921778371169026",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16508811191852041977017821887204137955816331040385276110261643892701458724933",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1804800467126006102677564831888710635194614232739335985819349312754063580223",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11189892034806587650995829160516587240879881493093022855087765921356611070470",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20567450145123179140729389574352706949280207113956641415022972885523439610844",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"4666756311257455192796774305229624459258864488677689058174087310651786875914",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11389253665835451896363091846189307652796786468610595637047377864063404843117",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18793736599347263150867965517898541872137378991464725717839931503944801692688",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"4206344588923325482680116848820594823631536459347642329098796888497153867720",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1739462481670645248707834504605096139894257554120906850613041004917967456145",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18514227342636266640333254638454588508118462110178719555586534011641424431745",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"17887039315911403193186866703775654467672391491657957999455462537283842145802",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"2824959020572825365047639014537190268717891749361604043531643698340708119767",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12521547103713919592301476538318318223836047611311454785951907894055964264287",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"8658146183671258251984364885894342376430874614261222570603159082682815800788",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"154390145585284450772861151318029820117470958184878116158462181541183085587",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7593705166056392393963956710828665339496927193740869686529339432486182720653",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"5529559239163081088908568555890212324771345012509269613465629182165427812002",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3729910453162885538930719732708124491456460687048972152311428493400220125686",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11942815243552870715777415109008273807076911177089425348095503288499102855779",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"498938524453430895689241565973888863905147713935369405079343247530256066618",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3976257517234324421403708035200810671331954932478384823208414346189926720724",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"723540703523219510043977323240437576248315561543814629392162302024056718473",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"13306548824219676333032339487546407241767961556934015003605485324283250885682",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7970147269291664639740298762956131361316495463191268382513594527221399186752",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20633313939958767604804835838065337107615699351647541991788258289962727735454",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17162090859520817529294904484646695645841022315617926715432606252643123848792",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"9181379842957190051440498041153333325098774266789773971685141362947015398641",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7051606617662816798224904133351061549832959857069896192072217769241273559278",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"16619522548478824222688310091434959542211899852679631815023615875678448806029",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"14965311177811968100298579672135357167599499478246106482433786066289128683961",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"9792733250919070275775594069208673385381167169182805600474820364274865306108",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"2069253833779081039049908513863485270550301879399727430830923273191877809560",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15847298987712771667136245955631872888473964330474501593909263901393348546986",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12244443532166430060291409356011430759892629145539185535677568234713942157668",
				)
				.unwrap(),
			]
			.to_vec(),
		]
		.to_vec();
	} else if i == 7 {
		return [
			[
				std::str::FromStr::from_str(
					"19332164824128329382868318451458022991369413618825711961282217322674570624669",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12346323761995603285640868741615937712088302657627126374070962894016296466118",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3913895681115272361294397190916803190924061797587910478563401817340941991811",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7048322889096718105055545382948709082135086733564574465991576956878202831861",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10375086910057323893637057154182902576957472442368661576421122036461645295833",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12765622911241487148932810040772504127756393086809438933166282251044289864727",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"266900212758702307861826326591090138389415348463003233900705815890364224151",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"14435131616556129905356866638030823183270286404767286105643513738132789033353",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"5780976801287540146775934937953368730928109502001687434229528186520268917700",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1618320442446662026869390273942730786145909339107736579759397243640902802126",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3818399583522206096165108192531271582827953520684743806492664825009577810261",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11764506724346386316602508039052965575734225646587104133777798242528580374987",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"2414215974836165993714858157462355581258152126063378817495129367240311967136",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17609437036230923129211608175600293197801044251801590649435913902851695334081",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"363438080029711424794236047863047716381155074181485245036621530063262917196",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"535766679023716739184211613469394818313893958493710642899297971974381051070",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"5305068908469731303772738758164870877638068032868328180355958394150421214337",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10807632568240507366657354568432178961148417327580695024415275247652313539292",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15964415873358391713354948903242729080763777490509563223190335273158191600135",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20700362719972015883260687302741075186857660623182772413609788566925949033885",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10135127975676256977820296631533839366076919827597067890970660746228807376456",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"4251490167543116819728642817282216847143714366441358372252125244838181656331",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7745587495915033527847242564710473705100826890903278244320948416581724663023",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11741113129223221800185946819924457344647035336264986754437921049066977440806",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11630296782890656599545188109639399768829653360050213193782325240600583381364",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16861140446185941149398487176581839232380972247302922484807333229513905651035",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"365879246117123675211400356410703684399715291171114630107795112994207447819",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21725607857580053522363567649763546934441685061337033780528788383243719579033",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"9222866548596464928765000608129177609426964853736257576074550520759533736918",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10261578281201197531384003420612639018011405529775212563256392340336951230146",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15644037447921591571869862919382888810859308861783088910843592577202362807673",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12752004188139535619565478547449108772137477456363099481095747591698702436636",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"4205805109630387448825516813913983509046636797101589615147198457314360427718",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21047095155106717901091873146599497621258071512562421967648909471775919992713",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15624165295872926124160584750951090817255240214488120310950503163805737026315",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"15064589937731741958666763896598138037875460434244947486199623542160035749721",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1801577872277160959016940766173040841160105238799805406938450020949902989173",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"2896766420608048344829901127120623317655260981420052771341833288256800199953",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12828791469509204618898135640019714232831708508424682785876476343251730674999",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21363471986981372923191391880511344708743312828234098289107697080824665183315",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21372706354350795416381912271616633829725494570576895047490974943034914894898",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16006531510217730955981102005088687858079561573088629102219485906666961331083",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"2389357602244845938251345005183369360523566673990464798041306722747500447645",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15275955107196234672088664710679934029171843237458844492987233368659104714648",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"8038797517535218686870517662905230585331773059774130312418943649247287196930",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17923922393436914864421862212181654800719733137689602673604754147078808030201",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12890519745320143484176500044628647247549456778462652469313611980363507314914",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"8058516556024397257577081553178859094042894928866720408652077334516681924252",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"768425396034382182896247252731538808045254601036758108993106260984310129743",
				)
				.unwrap(),
			]
			.to_vec(),
		]
		.to_vec();
	} else if i == 8 {
		return [
			[
				std::str::FromStr::from_str(
					"12051363189633051999486642007657476767332174247874678146882148540363198906151",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"6387692555402871022209406699166470377527846400909826148301704257996818597444",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"5501161701967897191598344153113501150221327945211106479845703139297020305204",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11704372055359680530622226011526065512090721245437046184430227296826364812961",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1448611482943320179763394986273491989368427112997509352702795612841455555221",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11429145481524962708631235759094055797723504985787912972575745356597208940857",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18021858528471759023192195347788820214752298716891162685115069036283008604659",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"19817577944622399780828745167469547332167999743980557486183403063955748437619",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"16868980302925985719076889965831700407328155411673408077166038059874616424216",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"14717432944340806781505761211058502775325970511884444497202848327581753493322",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"6273484270523289845253546319956998489830555038697388950038256377785540828355",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7726043103954429233325852791166106732104332590864071922310309250010129731951",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21052353119157611359715869265647287129868507410601603360127523286602350622783",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"14881796557136180514390287939887071460258251160875710427576954128871507002642",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16341327439981153879863707938117355436152690262312411284193970279829974799334",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10737675906107372302108775622264379258926415910493665638388971468924879578019",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"17652699767629314433191915267767147860052614073432922215674211498672835339113",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7457854400138129895665591719907473144796504905294990100367501377050420942800",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"2136850802972823585140870808569264373787409642804109426616292140046700710743",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"14029467347298896610468190615212519453678316548442709087191045978401072380889",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17927699952921266007590534383984238136710494507499176330493504416180410161683",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1404719213830610030709583332543456268094679432456284386108188509031502237811",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15774757292079018355173698870903422490868220545526384876021336136892926326596",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"13992040374687149195439840459922227749294794072303579532004750946306028893274",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"19895094843870397064274579657905921299619388074084417486420154568847155746891",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"943833985612967248618844364501030453998731991825395875139617731659343743483",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18334641092245356682448009823797080853859186519922476229272838591594967878678",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12440287044655505483131716236615633401781045711053210640202766768864619378050",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"19130942564098572936370308509908873069169152245172660555660369853346605570826",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"13687979327148217614616687417475244897906227789285703940171633508277844471062",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16887921327479880141959363366262254722342925451159884082370074726344024008329",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20378003125024698406589040864014894045124234695859352480989552885205935609512",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"9961553412530901953022991497331082655746860319830309417179972582392489275965",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17755268665220780466271147660314410613992814315871705414495724015443459797439",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15394131279964876131165951719955566821453162041574233072088124095626652523043",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12668230348320365182085867728169435383987570924921845106243310905832768752125",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"14046812111383844816383347755263287603387502282980410255379630204396960343368",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11590093969266595252327261214735156204516524792938909229175092594303424141199",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"4623517074925959322927421514289132524032863498392441375476410779446526502799",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"11550389531965919926150256242174358326491059727918559332939872696684299343135",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"408487396317981846281976563618407581852133413686169882346565860317912856432",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10717757571561029382519744040791773994731123262749372629687813122941078154016",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21323787615496251932181222397986048515693661833099659753170924658480548866921",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20780799310067873093555276926357624414275975377319941015818682052081980020892",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"9948385944800296129032348634683354181546876394979291412116493575442898426065",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"4957033413111065858035065225611730571499258914257595411830870977545212164095",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"5227254936689728148737265263965107718869714128941995977191096572191110991079",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3582814872786080867997255427740166393615552773099677831398251586195329933975",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"2136737803483410555580163900871515004623198990079556379647848364282254542316",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"2965752098571712086281180512370022839542603960309127077035724860894697782076",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1478525086510042909660572998242949118476342047444968703549274608283885678547",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3563375996604290844805064443647611841824012587505923250907062088840679700555",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15461452581843517997080348781604020486994675070532901120353124746087231692278",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20472517020063295821544268171575414161230806406668271887185150097779785573889",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21058001005918321995459971112208002381460494177332965873048074199074929946172",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15805746645980285645504697043988763170971539673993759868487715403982423015009",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"7141240965656437676130015766799708612940092856280620325870466265817008351948",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21418010338098024788434337801477243267248314524079104488811186206038748626642",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20272108634229595317682817969506273496034097230124371921628691470754475805838",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16734095147399743907618148751687506877774623133599770145304816136018239803101",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"8439324632051181834455499457268557602816180314723268640869118054114888151316",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"4953900961796661020464968131122569974581908893169105485631905994366550928492",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18071625983692455679240094911529791119099077429122520426399552756115503123111",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"19638917592063029281156873227053827678889868373299664608974791764751784473040",
				)
				.unwrap(),
			]
			.to_vec(),
		]
		.to_vec();
	} else if i == 9 {
		return [
			[
				std::str::FromStr::from_str(
					"708458300293891745856425423607721463509413916954480913172999113933455141974",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"14271228280974236486906321420750465147409060481575418066139408902283524749997",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15852878306984329426654933335929774834335684656381336212668681628835945610740",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"14650063533814858868677752931082459040894187001723054833238582599403791885108",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"5582010871038992135003913294240928881356211983701117708338786934614118892655",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17817167707934144056061336113828482446323869140602919022203233163412357573520",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16618894908063983272770489218670262360190849213687934219652137459014587794085",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10883405878649359800090160909097238327402403049670067541357916315880123123342",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7439184039942350631846254109167666628442833987137988596039526179738154790587",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"2727663760525187222746025175304386977552466570311228286110141668880678011929",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16992375884417886634716738306539629570444547136030480542879886913528563834233",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"4178586893949624406750122665277033849762243490544460031634329370298105635905",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"2517914797385699886738929430037355069462619900197972886482360691236776726214",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20164173810534657634631187494276970100735049909727379228976555863615716408280",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"19970958827248077001061220127605534603528515080207197493660642269195127427214",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15606275977308968307194602612931727810866183872589808138812916593200446820753",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12261436001550634140750381230737452634746867040398895669545077774504957433511",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10405309809257831434323731445544896504541938387524726028487604098725193737428",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"13408856444092113657034337770571899796129642125690066226794939383190876435468",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"19768080898957882918527124226120459667739640387901357739011662191034806046251",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16749889646056241484852997428132695501278739424507088920371060969471495213919",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12331609790192161246735870679870317366088443875784324655482358218146673901073",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15769331739277556832196167201116801527901089923090632364403958141614820528626",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"5227172275505968397128736045169568430462701766148126842874241545343535393924",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"919073378344729780131814412541912290691661039815032069498359347682919854836",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17858725475505870077023114050620337312678855554361132257763133392017321111169",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21805188450184460363143840112266872832328782034569970452376470141743078343745",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"15808413311863154368918155104905222670782553225279887458053980771135357021692",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"12828907214414139667587331812274388831051429093098655261887619166452245292431",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"19323880880917307340820066456419195877039970908109908221992925424585030574269",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17591732412986269470826282099678922890996647592922237928486497997144096433314",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"5282593184575641056912422403901924986019740793240905758215569065763629999318",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16013130707598525718519250412251656096494468043256226360413191733653074896117",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"928381583587170989315021718439506896903185927814675820160976165627097308915",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"13354336789663524324458402003354905134416094005220899335023797754517805691310",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"8780135673134081873589118311874067764073719549433574820315100541871522642766",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"3334957744389892864165113989538814646945861179021194859030934481494560681812",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10553413566358881045095498839713459314577909144176577153981801574128014927353",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18894321506279909207228932263261226433242541255661384643559047811974513999438",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20211894014628303327332299342564779073614790317614402383971270594430055013904",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16723480621932556506775906903415088312771104391224076734252099577243237899106",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1131872547334579236404174618548801749854242069301712398106619948805304881636",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17386814048141719093058723520379257085987299288710382497237609774141718421404",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"13729980537487612221640320393867198844745491357830417754869369043292518007370",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15860780436383591737179656321807464721751913977397035980422407138400867838633",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"14708550460111247278740231297332510059116901767161326454481923990389610737973",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3132820559166321299152015048428879769905404947939291493327190426785911502819",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"8658132367999084824971296219169212568783540935524918908332001856872807119287",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"21064783047501777742084787259676320053480170916619513986794406566953069418035",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20731000104011695148048713576219525164619502119638555785381543866326561323",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17189725817866212967650950297463469529475851286172280116066228706121595462088",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3310440878606659516028312898499559492876015493892608849966645073367377278233",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18463918215326370595980949760897480127622730018343709491036454088497976892863",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"10894192430593140913557164014343360386192963621862346779515699758352916852228",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"5060610877870389107953459328006060153180283860738879092399406248484265273634",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"9068988823145592214189961315730261367007076042069390630024839612151270430414",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"13160707893890865447331361630522644819624543031829773191665491273833460019183",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"13920568292534026180186486064598876780779571940988254327823480971820885713801",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3894011501178134026216736522445829906312115650019712122802932677318433032635",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"17895318821130376385979570244603067634449453259842805202694945793852667231847",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"9777993060458301797155055013115849176281006051494461044565335406558308324220",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16521293541516305251718414192107787058980727971856888501176820100904791554730",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"7744063601405355255689420547832904761861257642931934580021876189691881462544",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"5444730929053688962452159157646022068806222098484627080046464163159451208522",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1524118152994294864739915388438939180298324297960159419600850033701763764640",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"1334622237342346242862023763160346671504959163544406543315614662442562816653",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"16126317914306849967682996412350336172782726693375105190424151365140854833923",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"6345975085253358297751050638846919833013142450462810543971050115910612860460",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"2703875280053263252177031410407166981522153304496807669518295313468095058674",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"20550626512184448884716175825490086259235894802178999642552696391947509065676",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"15013718986700828670892638677446258841869291160144196138236407826511808592486",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"4682264015512203762723381542642871160915706748420642731100634327658667608042",
				)
				.unwrap(),
			]
			.to_vec(),
			[
				std::str::FromStr::from_str(
					"12834108073603507925748862283503586970613250684810871463629807392488566121352",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"8422606792378744850363509404165092879785007388646473871019846954536829739979",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"9339209090550177650528715604504958143078492516052997365409534971861874881780",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"9141831918422847136631159987994781722269889810731887947045878986971886716767",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"18329180549061748373684938917948729366786279119056979983310618862430068636631",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"2009551904565170718789964252583363785971078331314490170341991643087565227885",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"3859729780601667888281187160881197567257456581829833310753128034179061564519",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"8535335342372994336873304745903510543599314397287086554558824692658347277251",
				)
				.unwrap(),
				std::str::FromStr::from_str(
					"14148514289641991520153975838000398174635263164584825009402034843810351225518",
				)
				.unwrap(),
			]
			.to_vec(),
		]
		.to_vec();
	}

	return vec![];
}
