// Internal imports
use std::process::{ Command, Stdio };
use std::fs::{File, remove_file, remove_dir_all};
use std::io::Write;
use std::path::Path;
use std::time::Instant;

// External crates
use json::{object, array};
use structopt::StructOpt;
use colored::*;

#[derive(StructOpt)]
struct Config {
   #[structopt(parse(from_os_str))]
   target_files: Vec<std::path::PathBuf>,

   #[structopt(short = "v", long = "verbose")]
   verbose: bool,

   #[structopt(short = "d", long = "debug")]
   debug: bool,

   #[structopt(short = "s", long = "silent")]
   silent: bool,

   #[structopt(short = "a", long = "arguments")]
   execute_args: Vec<String>,
}

fn main() { let args = Config::from_args();
    build(&args);

    if args.debug {
       debug(&args);
    } else {
       run(&args);
    }
}

fn announce(announcement: &str, color: Option<Color>, verbose_announce: Option<bool>) {
    let verbose = verbose_announce.unwrap_or(false);
    let color = color.unwrap_or(Color::White);

    let line_start = if verbose { "--->".color(color).bold() } else { "----->".color(color).bold() };

    println!("{} {}", line_start, announcement.color(color).bold());
}

fn build(config: &Config) {
   announce("Compiling...", Some(Color::Blue), None);

   // Determine if file is C or C++
   let mut filepath: String = config.target_files[0].to_str().unwrap().to_string();
   let extension_offset = filepath.find('.').unwrap() + 1;
   let extension = filepath.split_off(extension_offset);

   let compiler: String = match extension.as_str() {
       "cpp" => "clang++".to_string(),
       "c" => "clang".to_string(),
       _ => {
           eprintln!("Source file must be either .c or .cpp");
           std::process::exit(-1);
       }
   };

   if config.verbose { announce("Detected filetype", None, Some(true)) }

   // Call compiler
   let now = Instant::now();

   let mut compile_command = Command::new(compiler);
   compile_command.args(config.target_files.clone());
   compile_command.args(&["-fsanitize=address", "-g", "-Wall", "-D DEVELOPMENT"]);
   compile_command.status().unwrap();

   if config.verbose { announce(format!("Compilation complete ({} ms)", now.elapsed().as_millis()).as_str(), None, Some(true)) }
}

fn run(config: &Config) {
   announce("Running...", Some(Color::Green), None);

   // Call executable
   let mut run_command = Command::new("./a.out");
   run_command.args(config.execute_args.clone());
   run_command.env("ASAN_OPTIONS=detect_leaks", "1");

   if config.silent {
      run_command.stdout(Stdio::null());
   }

   let now = Instant::now();
   let status = run_command.status().unwrap();
   if config.verbose { announce(format!("Program terminated with {} ({} ms)", status, now.elapsed().as_millis()).as_str(), None, Some(true)) }

   cleanup();
}

fn cleanup() {
   if Path::new("./a.out").exists() {
      remove_file("a.out").expect("Unable to delete executable");
   }

   if Path::new("./a.out.dSYM").exists() {
      remove_dir_all("a.out.dSYM").unwrap_or_default();
   }

   if Path::new("./.vimspector.json").exists() {
      remove_file(".vimspector.json").unwrap_or_default();
   }
}

fn debug(config: &Config) {
   if Path::new("./.vimspector.json").exists() {
      remove_file(".vimspector.json").expect("Unable to delete previous vimspector.json");
      std::process::exit(-1);
   }

   announce("Launching debugger...", Some(Color::Magenta), None);

   let debug_data: json::JsonValue = object!{
      "configurations": json::object!{
         "lldb-debug": json::object!{
            "adapter": "vscode-cpptools",
            "configuration": {
               "request": "launch",
               "program": "${workspaceRoot}/a.out",
               "args": config.execute_args.clone(),
               "cwd": "${workspaceRoot}",
               "environment": array![],
               "stopAtEntry": true,
               "MIMode": "lldb",
            }
         }
      }
   };

   let vimspector_path = Path::new(".vimspector.json");
   let mut file = match File::create(&vimspector_path) {
      Err(why) => {
         eprintln!("couldn't create {}: {}", vimspector_path.display(), why);
         std::process::exit(-1);
      },
     Ok(file) => file,
    };

   if config.verbose { announce("Created .vimspector.json", None, Some(true)) }

   file.write_all(json::stringify(debug_data).as_bytes()).expect("Failed to write to .vimspector.json");

    if config.verbose { announce("Wrote .vimspector.json", None, Some(true)) }

   let mut vim_command = Command::new("nvim");
   let now = Instant::now();

   vim_command.status().unwrap();
   if config.verbose { announce(format!("Debugging complete ({} ms)", now.elapsed().as_millis()).as_str(), None, Some(true)) }

   cleanup();
}
