use crate::{GithubClient, Repository, SortDirection, User};
use anyhow::{bail, Context, Result};
use async_trait::async_trait;
use jacklog::{debug, trace};
use serde::Deserialize;

#[derive(Debug, PartialEq, Eq, Deserialize)]
pub struct Team {
    pub id: u64,
    pub node_id: String,
    pub url: String,
    pub html_url: String,
    pub name: String,
    pub slug: String,
    pub description: String,
    pub privacy: String,
    pub permission: String,
    pub members_count: Option<u64>,
    pub repos_count: Option<u64>,
}

#[derive(Debug)]
pub enum TeamRole {
    Member,
    Maintainer,
    All,
}

#[derive(Default, Debug)]
pub struct GetTeamByNameRequest {
    pub org: String,
    pub team_slug: String,
}

#[derive(Debug)]
pub struct GetTeamByNameResponse {
    pub team: Team,
}

#[derive(Default, Debug)]
pub struct ListTeamRepositoriesRequest {
    pub org: String,
    pub team_slug: String,
    pub per_page: Option<usize>,
    pub page: Option<usize>,
}

#[derive(Debug)]
pub struct ListTeamRepositoriesResponse {
    pub repositories: Vec<Repository>,
}

#[derive(Default, Debug)]
pub struct ListTeamMembersRequest {
    pub org: String,
    pub team_slug: String,
    pub role: Option<TeamRole>,
    pub per_page: Option<usize>,
    pub page: Option<usize>,
}

#[derive(Debug)]
pub struct ListTeamMembersResponse {
    pub members: Vec<User>,
}

#[async_trait]
pub trait Teams {
    async fn get_team_by_name(&self, input: GetTeamByNameRequest) -> Result<GetTeamByNameResponse>;
    async fn list_team_repositories(
        &self,
        input: ListTeamRepositoriesRequest,
    ) -> Result<ListTeamRepositoriesResponse>;
    async fn list_team_members(
        &self,
        input: ListTeamMembersRequest,
    ) -> Result<ListTeamMembersResponse>;
}

#[async_trait]
impl Teams for GithubClient {
    async fn get_team_by_name(&self, input: GetTeamByNameRequest) -> Result<GetTeamByNameResponse> {
        // Make the request.
        let res = self
            .client()
            .get(&format!(
                "https://api.github.com/orgs/{}/teams/{}",
                input.org, input.team_slug
            ))
            .send()
            .await?;

        // Check the response code.
        if !res.status().is_success() {
            bail!("{}", res.status().canonical_reason().unwrap_or(&"unknown"));
        }

        // Get the response text so we can dump it for debugging.
        let res = res.text().await?;
        trace!("{}", &res);

        //let res: Response = res.json().await?;
        let team: Team =
            serde_json::from_str(&res).context("error parsing Team from get_team_by_name")?;
        debug!("{:?}", &team);

        Ok(GetTeamByNameResponse { team })
    }

    async fn list_team_repositories(
        &self,
        input: ListTeamRepositoriesRequest,
    ) -> Result<ListTeamRepositoriesResponse> {
        // Make the request.
        let res = self
            .client()
            .get(&format!(
                "https://api.github.com/orgs/{}/teams/{}/repos",
                input.org, input.team_slug
            ))
            .send()
            .await?;

        // Check the response code.
        if !res.status().is_success() {
            bail!("{}", res.status().canonical_reason().unwrap_or(&"unknown"));
        }

        // Get the response text so we can dump it for debugging.
        let res = res.text().await?;
        trace!("{}", &res);

        //let res: Response = res.json().await?;
        let repositories = serde_json::from_str(&res)
            .context("error parsing response from list_team_repositories")?;
        debug!("{:?}", &repositories);

        Ok(ListTeamRepositoriesResponse { repositories })
    }

    async fn list_team_members(
        &self,
        input: ListTeamMembersRequest,
    ) -> Result<ListTeamMembersResponse> {
        // Make the request.
        let res = self
            .client()
            .get(&format!(
                "https://api.github.com/orgs/{}/teams/{}/members",
                input.org, input.team_slug
            ))
            .send()
            .await?;

        // Check the response code.
        if !res.status().is_success() {
            bail!("{}", res.status().canonical_reason().unwrap_or(&"unknown"));
        }

        // Get the response text so we can dump it for debugging.
        let res = res.text().await?;
        trace!("{}", &res);

        //let res: Response = res.json().await?;
        let members =
            serde_json::from_str(&res).context("error parsing response from list_team_members")?;
        debug!("{:?}", &members);

        Ok(ListTeamMembersResponse { members })
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::{client, handle, org, repo};
    use std::env;

    #[cfg(test)]
    fn team() -> String {
        env::var("GITHUB_TEST_TEAM_SLUG").unwrap()
    }

    #[tokio::test]
    async fn test_get_team_by_name() {
        let client = client();
        let res = client
            .get_team_by_name(GetTeamByNameRequest {
                org: org(),
                team_slug: team(),
            })
            .await
            .unwrap();

        assert_eq!(res.team.slug, team());
    }

    #[tokio::test]
    async fn test_list_team_repositories() {
        let client = client();
        let res = client
            .list_team_repositories(ListTeamRepositoriesRequest {
                org: org(),
                team_slug: team(),
                ..Default::default()
            })
            .await
            .unwrap();

        assert_eq!(res.repositories.first().unwrap().name, repo());
    }

    #[tokio::test]
    async fn test_list_team_members() {
        let client = client();
        let res = client
            .list_team_members(ListTeamMembersRequest {
                org: org(),
                team_slug: team(),
                ..Default::default()
            })
            .await
            .unwrap();

        // Make sure we can find our user. This (like most tests) requires that
        // the user be a member of the test group.
        res.members.iter().find(|u| u.login == handle()).unwrap();
    }
}
