use crate::{GithubClient, SortDirection, User};
use anyhow::{bail, format_err, Context, Result};
use async_trait::async_trait;
use jacklog::{debug, trace};
use reqwest::header::LINK;
use serde::{Deserialize, Serialize};
use std::path::PathBuf;

#[derive(Debug, Deserialize, Serialize)]
pub struct Commit {
    pub author: GitUser,
    pub committer: GitUser,
    pub message: String,
    pub tree: Tree,
    pub comment_count: usize,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct GitUser {
    pub name: String,
    pub email: String,
    pub date: String,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Tree {
    pub sha: String,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Stats {
    pub additions: usize,
    pub deletions: usize,
    pub total: usize,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct File {
    pub filename: String,
    pub additions: usize,
    pub deletions: usize,
    pub changes: usize,
    pub status: String,
    pub raw_url: String,
    pub blob_url: String,
    pub patch: Option<String>,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Repository {
    pub id: u64,
    pub node_id: String,
    pub name: String,
    pub full_name: String,
    pub owner: User,
    pub private: bool,
    pub html_url: String,
}

#[derive(Debug, Default, Deserialize, Serialize)]
pub struct ListRepositoriesResponse {
    pub repositories: Vec<Repository>,
    pub next: Option<String>,
    pub last: Option<String>,
    pub first: Option<String>,
    pub prev: Option<String>,
}

#[derive(Default, Debug)]
pub struct ListRepositoriesRequest {
    pub org: String,
    pub _type: Option<RepositoryType>,
    pub sort: Option<ListRepositoriesSort>,
    pub direction: Option<SortDirection>,
    pub per_page: Option<usize>, // max 100
    pub page: Option<usize>,
}

#[derive(Default, Debug)]
pub struct GetCommitRequest {
    pub owner: String,
    pub repo: String,
    pub _ref: String,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct GetCommitResponse {
    pub sha: String,
    pub commit: Commit,
    pub author: Option<User>,
    pub committer: User,
    pub parents: Vec<Tree>,
    pub stats: Stats,
    pub files: Vec<File>,
    pub next: Option<String>,
    pub last: Option<String>,
    pub first: Option<String>,
    pub prev: Option<String>,
}

#[derive(Default, Debug)]
pub struct GetRepositoryContentRequest {
    pub owner: String,
    pub repo: String,
    pub path: PathBuf,
    /// Defaults to the repository's default branch.
    pub _ref: Option<String>,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct GetRepositoryContentResponse {
    #[serde(rename = "type")]
    pub _type: ContentType,
    pub encoding: String,
    pub size: u64,
    pub name: String,
    pub path: PathBuf,
    pub content: String,
    pub sha: String,
    pub html_url: String,
    pub download_url: String,
    pub next: Option<String>,
}

#[derive(Deserialize, Serialize, Debug)]
#[serde(rename_all = "lowercase")]
pub enum ContentType {
    File,
    Symlink,
}

#[derive(Debug)]
pub enum RepositoryType {
    All,
    Public,
    Private,
    Forks,
    Sources,
    Member,
    Internal,
}

#[derive(Debug)]
pub enum ListRepositoriesSort {
    Created,
    Updated,
    Pushed,
    FullName,
}

#[async_trait]
pub trait Repositories {
    async fn list_repositories(
        &self,
        input: ListRepositoriesRequest,
    ) -> Result<ListRepositoriesResponse>;

    async fn get_commit(&self, input: GetCommitRequest) -> Result<GetCommitResponse>;

    async fn get_repository_content(
        &self,
        input: GetRepositoryContentRequest,
    ) -> Result<GetRepositoryContentResponse>;
}

#[async_trait]
impl Repositories for GithubClient {
    async fn list_repositories(
        &self,
        input: ListRepositoriesRequest,
    ) -> Result<ListRepositoriesResponse> {
        // Make the request.
        let res = self
            .client()
            .get(&format!("https://api.github.com/orgs/{}/repos", input.org,))
            .send()
            .await?;

        // Check the response code.
        if !res.status().is_success() {
            bail!("{}", res.status().canonical_reason().unwrap_or(&"unknown"));
        }

        // Get the link metadata.
        //let link = res.headers().get("link").unwrap().to_str()?.to_string();

        // Get the response text so we can dump it for debugging.
        let res = res.text().await?;
        trace!("{}", &res);

        //let res: Response = res.json().await?;
        let repositories: Vec<Repository> = serde_json::from_str(&res)?;
        debug!("{:?}", &repositories);

        Ok(ListRepositoriesResponse {
            repositories,
            //next: Some(link),
            ..Default::default()
        })
    }

    /// Get a commit from a repository.
    async fn get_commit(&self, input: GetCommitRequest) -> Result<GetCommitResponse> {
        // Make the request.
        let res = self
            .client()
            .get(&format!(
                "https://api.github.com/repos/{}/{}/commits/{}",
                input.owner, input.repo, input._ref
            ))
            .send()
            .await?;

        // Check the response code.
        if !res.status().is_success() {
            bail!(
                "buhtig: get_commit: error making request: {}",
                res.status().canonical_reason().unwrap_or(&"unknown")
            );
        }

        // Get the link metadata.
        let link = res
            .headers()
            .get("link")
            .map(|l| l.to_str().unwrap().to_string());

        // Get the response text so we can dump it for debugging.
        let res = res.text().await.context("getting text from request")?;
        trace!("{:?}", &res);

        //let res: Response = res.json().await?;
        let mut res: GetCommitResponse =
            serde_json::from_str(&res).context("parse GetCommitResponse")?;
        debug!("{:?}", &res);

        // Add the LINK metadata.
        res.next = link;

        Ok(res)
    }

    /// Get the content of a path at a given commit.
    ///
    /// TODO: Support getting directories in addition to files and symlinks.
    async fn get_repository_content(
        &self,
        input: GetRepositoryContentRequest,
    ) -> Result<GetRepositoryContentResponse> {
        // Make the request.
        let res = self
            .client()
            .get(&format!(
                "https://api.github.com/repos/{}/{}/contents/{}",
                input.owner,
                input.repo,
                input
                    .path
                    .as_path()
                    .to_str()
                    .ok_or(format_err!("path is not unicode"))?,
            ))
            .send()
            .await?;

        // Check the response code.
        if !res.status().is_success() {
            bail!("{}", res.status().canonical_reason().unwrap_or(&"unknown"));
        }

        // Get the link metadata.
        let link = res
            .headers()
            .get("link")
            .map(|l| l.to_str().unwrap().to_string());

        // Get the response text so we can dump it for debugging.
        let res = res.text().await?;
        trace!("{}", &res);

        //let res: Response = res.json().await?;
        let mut res: GetRepositoryContentResponse = serde_json::from_str(&res)?;
        debug!("{:?}", &res);

        // Add the LINK metadata.
        res.next = link;

        Ok(res)
    }
}

#[cfg(test)]
mod tests {
    use std::str::FromStr;

    use super::*;
    use crate::EnvironmentProvider;
    use crate::{org, repo};

    #[tokio::test]
    async fn test_list_repositories() {
        let client = GithubClient::new(&EnvironmentProvider::default()).unwrap();
        let res = client
            .list_repositories(ListRepositoriesRequest {
                org: org(),
                per_page: Some(1),
                ..Default::default()
            })
            .await
            .unwrap();

        // Check response.
        assert_eq!(res.repositories.first().unwrap().name, repo());

        // Check headers.
        //assert_eq!(res.next.unwrap(), "");
    }

    #[tokio::test]
    async fn test_get_repository_content() {
        let client = GithubClient::new(&EnvironmentProvider::default()).unwrap();
        let res = client
            .get_repository_content(GetRepositoryContentRequest {
                owner: org(),
                repo: repo(),
                path: PathBuf::from_str("foobar.txt").unwrap(),
                ..Default::default()
            })
            .await
            .unwrap();

        // Check response.
        let actual = base64::decode(&res.content.trim()).unwrap();
        assert_eq!(actual, "foobaz\n".as_bytes());
        assert_eq!(
            res.sha,
            "9dabfec9194affdefddeffab8a74ec018554bf56".to_string()
        );
    }
}

/*
curl -v \
  -H "Authorization: Bearer $GITHUB_TOKEN" \
  -H "Accept: application/vnd.github.v3+json" \
  'https://api.github.com/orgs/remind101/teams/r2d2/repos?per_page=1'
*/
