use crate::CredentialsProvider;
use anyhow::Result;
use jacklog::debug;
use reqwest::{
    header::HeaderMap,
    header::{ACCEPT, AUTHORIZATION, USER_AGENT},
    Client,
};
use std::fmt::Debug;

#[derive(Debug)]
pub struct GithubClient {
    client: Client,
}

impl GithubClient {
    /// Create a new `GithubClient` using a credentials provider.
    ///
    /// # Errors
    ///
    /// Returns an error parsing credentials or configuring the HTTP client.
    pub fn new<T: CredentialsProvider + Debug>(credentials: &T) -> Result<Self> {
        let mut headers = HeaderMap::new();
        headers.insert(USER_AGENT, "buhtig".parse()?);
        headers.insert(ACCEPT, "application/vnd.github.v3+json".parse()?);
        headers.insert(
            AUTHORIZATION,
            format!("Bearer {}", credentials.token()).parse()?,
        );
        let client = Client::builder().default_headers(headers).build()?;
        debug!("{:#?}", &client);

        Ok(Self { client })
    }

    pub(crate) fn client(&self) -> &Client {
        &self.client
    }
}
