use anyhow::Result;
use buhtig::{
    EnvironmentProvider, GetTeamByNameRequest, GithubClient, ListPullRequestsRequest,
    ListReviewsForPullRequestRequest, ListTeamMembersRequest, ListTeamRepositoriesRequest,
    PullRequestState, Pulls, Repositories, Teams,
};
use std::env;

#[tokio::main]
async fn main() -> Result<()> {
    jacklog::init(Some(&"warn".to_string()))?;

    let org = env::var("GITHUB_ORG")?;
    let team_slug = env::var("GITHUB_TEAM_SLUG")?;

    let client = GithubClient::new(&EnvironmentProvider::default())?;

    let team = client
        .get_team_by_name(GetTeamByNameRequest {
            org: org.clone(),
            team_slug: team_slug.clone(),
        })
        .await?
        .team;

    // Get all the members of the team.
    let members = client
        .list_team_members(ListTeamMembersRequest {
            org: org.clone(),
            team_slug: team.slug.clone(),
            per_page: Some(100),
            ..Default::default()
        })
        .await?
        .members;

    let repos = client
        .list_team_repositories(ListTeamRepositoriesRequest {
            org: org.clone(),
            team_slug: team.slug,
            per_page: Some(100),
            ..Default::default()
        })
        .await?
        .repositories;

    let mut out = vec![];
    for repo in repos {
        let prs = client
            .list_pull_requests(ListPullRequestsRequest {
                owner: repo.owner.login.clone(),
                repo: repo.name.clone(),
                state: Some(PullRequestState::Open),
                per_page: Some(100),
                ..Default::default()
            })
            .await?
            .pull_requests;

        for pr in prs {
            let reviews = client
                .list_reviews_for_pull_request(ListReviewsForPullRequestRequest {
                    owner: repo.owner.login.clone(),
                    repo: repo.name.clone(),
                    pull_number: pr.number,
                    per_page: Some(100),
                    ..Default::default()
                })
                .await?
                .reviews;

            // Go through each review. If the review is from a team member, then
            // don't output this review.
            if reviews
                .iter()
                .any(|r| members.iter().find(|m| m.login == r.user.login).is_some())
            {
                continue;
            }

            // If we got this far, it means no team members have reviewed this
            // PR, so print it out.
            out.push(format!(
                "{}\t{}\t{}\t{:15}\t{:72}\t{}",
                pr.created_at,
                pr.id,
                pr.state,
                &pr.user.login,
                &pr.title,
                reviews
                    .iter()
                    .map(|r| format!("{}:{}", r.user.login, r.state))
                    .collect::<Vec<String>>()
                    .join(", "),
            ));
        }
    }

    for line in out {
        println!("{}", line);
    }

    Ok(())
}
