use crate::{GithubClient, SortDirection, Team, User};
use anyhow::{bail, Context, Result};
use async_trait::async_trait;
use jacklog::{debug, trace};
use serde::Deserialize;

#[derive(Debug, Eq, PartialEq, Deserialize)]
pub struct PullRequest {
    pub url: String,
    pub id: u64,
    pub node_id: String,
    pub html_url: String,
    pub diff_url: String,
    pub patch_url: String,
    pub number: u64,
    pub state: String,
    pub locked: bool,
    pub title: String,
    pub user: User,
    pub body: String,
    pub labels: Vec<Label>,
    pub created_at: String,
    pub updated_at: String,
    pub closed_at: Option<String>,
    pub assignee: Option<User>,
    pub assignees: Vec<User>,
    pub requested_reviewers: Vec<User>,
    pub requested_teams: Vec<Team>,
    pub author_association: String,
    pub draft: bool,
}

#[derive(Debug, Deserialize)]
pub enum PullRequestState {
    Open,
    Closed,
    All,
}

#[derive(Debug, PartialEq, Eq, Deserialize)]
pub struct Label {
    pub id: u64,
    pub node_id: String,
    pub url: String,
    pub name: String,
    pub description: Option<String>,
    pub color: String,
    pub default: bool,
}

#[derive(Debug, PartialEq, Eq, Deserialize)]
pub struct Review {
    pub id: u64,
    pub node_id: String,
    pub user: User,
    pub body: String,
    pub state: String,
    pub submitted_at: String,
    pub commit_id: String,
    pub author_association: String,
}

#[derive(Debug, Deserialize)]
pub enum ListPullRequestsSort {
    Created,
    Updated,
    Popularity,
    #[serde(rename = "long-running")]
    LongRunning,
}

#[derive(Default, Debug)]
pub struct ListPullRequestsRequest {
    pub owner: String,
    pub repo: String,
    pub state: Option<PullRequestState>,
    pub head: Option<String>,
    pub base: Option<String>,
    pub sort: Option<ListPullRequestsSort>,
    pub direction: Option<SortDirection>,
    pub per_page: Option<usize>, // max 100
    pub page: Option<usize>,
}

#[derive(Debug)]
pub struct ListPullRequestsResponse {
    pub pull_requests: Vec<PullRequest>,
}

#[derive(Default, Debug)]
pub struct ListReviewsForPullRequestRequest {
    pub owner: String,
    pub repo: String,
    pub pull_number: u64,
    pub per_page: Option<usize>, // max 100
    pub page: Option<usize>,
}

#[derive(Debug)]
pub struct ListReviewsForPullRequestResponse {
    pub reviews: Vec<Review>,
}

#[async_trait]
pub trait Pulls {
    async fn list_pull_requests(
        &self,
        input: ListPullRequestsRequest,
    ) -> Result<ListPullRequestsResponse>;

    async fn list_reviews_for_pull_request(
        &self,
        input: ListReviewsForPullRequestRequest,
    ) -> Result<ListReviewsForPullRequestResponse>;
}

#[async_trait]
impl Pulls for GithubClient {
    async fn list_pull_requests(
        &self,
        input: ListPullRequestsRequest,
    ) -> Result<ListPullRequestsResponse> {
        // Make the request.
        let res = self
            .client()
            .get(&format!(
                "https://api.github.com/repos/{}/{}/pulls",
                input.owner, input.repo
            ))
            .send()
            .await?;

        // Check the response code.
        if !res.status().is_success() {
            bail!("{}", res.status().canonical_reason().unwrap_or(&"unknown"));
        }

        // Get the response text so we can dump it for debugging.
        let res = res.text().await?;
        trace!("{}", &res);
        std::fs::write("/tmp/res.json", &res)?;

        //let res: Response = res.json().await?;
        let pull_requests =
            serde_json::from_str(&res).context("error parsing results from list_pull_requests")?;
        debug!("{:?}", &pull_requests);

        Ok(ListPullRequestsResponse { pull_requests })
    }

    async fn list_reviews_for_pull_request(
        &self,
        input: ListReviewsForPullRequestRequest,
    ) -> Result<ListReviewsForPullRequestResponse> {
        // Make the request.
        let res = self
            .client()
            .get(&format!(
                "https://api.github.com/repos/{}/{}/pulls/{}/reviews",
                input.owner, input.repo, input.pull_number
            ))
            .send()
            .await?;

        // Check the response code.
        if !res.status().is_success() {
            bail!("{}", res.status().canonical_reason().unwrap_or(&"unknown"));
        }

        // Get the response text so we can dump it for debugging.
        let res = res.text().await?;
        trace!("{}", &res);
        std::fs::write("/tmp/res.json", &res)?;

        //let res: Response = res.json().await?;
        let reviews = serde_json::from_str(&res)
            .context("error parsing reviews from list_reviews_for_pull_request")?;
        debug!("{:?}", &reviews);

        Ok(ListReviewsForPullRequestResponse { reviews })
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::{client, org, repo};
    use std::env;

    #[cfg(test)]
    fn team() -> String {
        env::var("GITHUB_TEST_TEAM_SLUG").expect("error reading GITHUB_TEST_TEAM_SLUG")
    }

    #[tokio::test]
    async fn test_list_pull_requests() {
        let client = client();
        let res = client
            .list_pull_requests(ListPullRequestsRequest {
                owner: org(),
                repo: repo(),
                ..Default::default()
            })
            .await
            .unwrap();

        assert_eq!(res.pull_requests, vec![]);
    }
}
