use crate::params::Params;

pub(crate) struct Logger {
    daemon: bool,
    level: log::LevelFilter,
}

static mut LOGGER: Logger = Logger {
    daemon: false,
    level: log::LevelFilter::Info,
};

impl Logger {
    fn now(&self) -> String {
        if self.daemon {
            chrono::Local::now()
                .format("%Y-%m-%d %H:%M:%S: ")
                .to_string()
        } else {
            String::new()
        }
    }

    pub(crate) fn init(p: &Params) -> Result<(), log::SetLoggerError> {
        unsafe {
            LOGGER.daemon = p.daemon;
            LOGGER.level = p.log_level;
            log::set_logger(&LOGGER).map(|()| log::set_max_level(p.log_level))
        }
    }
}

impl log::Log for Logger {
    fn enabled(&self, metadata: &log::Metadata) -> bool {
        metadata.level() <= self.level
    }

    fn log(&self, record: &log::Record) {
        if self.enabled(record.metadata()) {
            match record.level() {
                log::Level::Warn | log::Level::Error => {
                    eprintln!("{}{}: {}", self.now(), record.level(), record.args())
                }
                _ => println!("{}{}: {}", self.now(), record.level(), record.args()),
            }
        }
    }

    fn flush(&self) {}
}
