/// structure representing a progressbar
#[derive(Clone)]
pub struct ProgressBar {
    progress: u8,
    text: String,
    empty: char,
    full: char,
}

impl ProgressBar {
    /// create a new progress bar
    /// # Arguments
    /// *`text`: the text to use for the progress bar
    /// *`empty`: the character to use for empty progress bar fields
    /// *`full`: the character to use for full progress bar fields
    /// # Returns
    /// Returns a new progress bar
    pub fn new(text: &str, empty: char, full: char) -> ProgressBar {
        ProgressBar {
            progress: 0u8,
            text: text.to_string(),
            empty,
            full,
        }
    }

    /// increase the current progress
    pub fn incr(&mut self) {
        if self.progress < 100u8 {
            self.progress += 1u8;
        }
    }

    /// set the current progress
    pub fn set_progress(&mut self, amount: u8) {
        if amount > 100u8 {
            self.progress = 100u8;
        } else {
            self.progress = amount;
        }
    }
    /// render the current progress bar
    pub fn render(&self) -> String {
        let mut bar: String = String::new();
        let mut idx: u8 = 0;
        while idx != self.progress {
            bar.push_str(format!("{}", self.full).as_str());
            idx += 1;
        }
        let leftover: u8 = 100u8 - self.progress;
        idx = 0;
        while idx != leftover {
            bar.push_str(format!("{}", self.empty).as_str());
            idx += 1;
        }
        format!("{}: [{}] {}%", self.text.clone(), bar, self.progress)
    }
}

/// struct representing a progressbar with extra features
#[derive(Clone)]
pub struct ExtProgressBar {
    format: String,
    text: String,
    progress: u8,
}

impl ExtProgressBar {
    /// create a new progress bar
    pub fn new(format: &str, text: &str) -> ExtProgressBar {
        ExtProgressBar {
            format: format.to_string(),
            text: text.to_string(),
            progress: 0,
        }
    }

    /// increase current progress by one
    pub fn incr(&mut self) {
        if self.progress != 100 {
            self.progress += 1;
        }
    }

    /// set the progress to a specific amount
    /// # Arguments
    /// *`amount`: the amount to set
    pub fn set_progress(&mut self, amount: u8) {
        if amount > 100 {
            self.progress = 100;
        } else {
            self.progress = amount;
        }
    }

    /// render the progress bar to a printable string
    pub fn render(&self) -> String {
        match self.format.len() {
            4 => {
                let mut format = self.format.chars();

                let begin: char = format.next().unwrap();
                let full: char = format.next().unwrap();
                let empty: char = format.next().unwrap();
                let close: char = format.next().unwrap();

                let mut bar: String = String::new();
                let mut idx: u8 = 0;
                while idx != self.progress {
                    bar.push_str(format!("{}", full).as_str());
                    idx += 1;
                }

                idx = 0;
                let leftover: u8 = 100 - self.progress;
                while idx != leftover {
                    bar.push_str(format!("{}", empty).as_str());
                    idx += 1;
                }
                format!(
                    "{}: {}{}{} {}%",
                    self.text, begin, bar, close, self.progress
                )
            }
            5 => {
                let mut format = self.format.chars();

                let begin: char = format.next().unwrap();
                let full: char = format.next().unwrap();
                let full_end: char = format.next().unwrap();
                let empty: char = format.next().unwrap();
                let close: char = format.next().unwrap();

                let mut bar: String = String::new();
                let mut idx: u8 = 0;
                while idx != self.progress {
                    if (self.progress - idx) == 1 {
                        bar.push(full_end);
                    } else {
                        bar.push(full);
                    }
                    idx += 1;
                }

                idx = 0;
                let leftover: u8 = 100 - self.progress;
                while idx != leftover {
                    bar.push(empty);
                    idx += 1;
                }
                format!(
                    "{}: {}{}{} {}%",
                    self.text, begin, bar, close, self.progress
                )
            }

            _ => {
                let mut bar: String = String::new();
                let mut idx: u8 = 0;
                while idx != self.progress {
                    if (self.progress - idx) == 1 {
                        bar.push('>');
                    } else {
                        bar.push('=');
                    }
                    idx += 1;
                }

                idx = 0;
                let leftover: u8 = 100 - self.progress;
                while idx != leftover {
                    bar.push(' ');
                    idx += 1;
                }
                format!("{}: [{}] {}%", self.text, bar, self.progress)
            }
        }
    }
}
