/// structure representing a progressbar
#[derive(Clone)]
pub struct ProgressBar {
    progress: f32,
    text: String,
    empty: char,
    full: char,
}

impl ProgressBar {
    /// create a new progress bar
    /// # Arguments
    /// *`text`: the text to use for the progress bar
    /// *`empty`: the character to use for empty progress bar fields
    /// *`full`: the character to use for full progress bar fields
    /// # Returns
    /// Returns a new progress bar
    pub fn new(text: &str, empty: char, full: char) -> ProgressBar {
        ProgressBar {
            progress: 0.0,
            text: text.to_string(),
            empty,
            full,
        }
    }

    /// increase the current progress
    pub fn incr(&mut self) {
        if self.progress < 100.0 {
            self.progress += 1.0;
        }
    }

    /// set the current progress
    pub fn set_progress(&mut self, amount: f32) {
        if amount > 100.0 {
            self.progress = 100.0;
        } else {
            self.progress = amount;
        }
    }
    /// render the current progress bar
    pub fn render(&self) -> String {
        let mut bar: String = String::new();
        let mut idx: u8 = 0;
        while (idx as f32) < self.progress {
            bar.push_str(format!("{}", self.full).as_str());
            idx += 1;
        }
        let leftover: u8 = (100.0 - self.progress).floor() as u8;
        idx = 0;
        while idx != leftover {
            bar.push_str(format!("{}", self.empty).as_str());
            idx += 1;
        }
        format!("{}: [{}] {:.2}%", self.text.clone(), bar, self.progress)
    }
}

/// struct representing a progressbar with extra features
#[derive(Clone)]
pub struct ExtProgressBar {
    format: String,
    text: String,
    progress: f32,
}

impl ExtProgressBar {
    /// create a new progress bar
    pub fn new(format: &str, text: &str) -> ExtProgressBar {
        ExtProgressBar {
            format: format.to_string(),
            text: text.to_string(),
            progress: 0.0,
        }
    }

    /// increase current progress by one
    pub fn incr(&mut self) {
        if self.progress < 100.0 {
            self.progress += 1.0;
        }
    }

    /// set the progress to a specific amount
    /// # Arguments
    /// *`amount`: the amount to set
    pub fn set_progress(&mut self, amount: f32) {
        if amount > 100.0 {
            self.progress = 100.0;
        } else {
            self.progress = amount;
        }
    }

    /// render the progress bar to a printable string
    pub fn render(&self) -> String {
        match self.format.len() {
            4 => {
                let mut format = self.format.chars();

                let begin: char = format.next().unwrap();
                let full: char = format.next().unwrap();
                let empty: char = format.next().unwrap();
                let close: char = format.next().unwrap();

                let mut bar: String = String::new();
                let mut idx: u8 = 0;
                while (idx as f32) < self.progress {
                    bar.push_str(format!("{}", full).as_str());
                    idx += 1;
                }

                idx = 0;
                let leftover: u8 = (100.0 - self.progress) as u8;
                while idx != leftover {
                    bar.push_str(format!("{}", empty).as_str());
                    idx += 1;
                }
                format!(
                    "{}: {}{}{} {:.2}%",
                    self.text, begin, bar, close, self.progress
                )
            }
            5 => {
                let mut format = self.format.chars();

                let begin: char = format.next().unwrap();
                let full: char = format.next().unwrap();
                let full_end: char = format.next().unwrap();
                let empty: char = format.next().unwrap();
                let close: char = format.next().unwrap();

                let mut bar: String = String::new();
                let mut idx: u8 = 0;
                while (idx as f32) < self.progress {
                    if ((self.progress - idx as f32).floor() - 1.0).abs() < 1.0 {
                        bar.push(full_end);
                        break;
                    } else {
                        bar.push(full);
                    }
                    idx += 1;
                }

                idx = 0;
                let leftover: u8 = (100.0 - self.progress).floor() as u8;
                while idx != leftover {
                    bar.push(empty);
                    idx += 1;
                }
                format!(
                    "{}: {}{}{} {:.2}%",
                    self.text, begin, bar, close, self.progress
                )
            }

            _ => {
                let mut bar: String = String::new();
                let mut idx: u8 = 0;
                while (idx as f32) < self.progress {
                    if ((self.progress - idx as f32).floor() - 1.0).abs() < 1.0 {
                        bar.push('>');
                    } else {
                        bar.push('=');
                    }
                    idx += 1;
                }

                idx = 0;
                let leftover: u8 = (100.0 - self.progress).floor() as u8;
                while idx != leftover {
                    bar.push(' ');
                    idx += 1;
                }
                format!("{}: [{}] {:.2}%", self.text, bar, self.progress)
            }
        }
    }
}
