use std::cmp::Ordering;
use std::io::{stderr, stdin, stdout};
use std::io::{Error, ErrorKind, Write};
use std::io::{Stderr, Stdin, Stdout};
use std::ops::Neg;
use crate::widget;
/// module containig different console actions for linux
#[deprecated(
    since = "0.6.7",
    note = "all functionality of console is now in a Terminal struct"
)]
pub mod console {
    /// enum for display control
    pub enum DisplayControl {
        /// clear the display (doesnt move cursor)
        Clear,
        /// clear the current line the cursor is on (doesn't move the cursor)
        ClearLine,
        /// reset the display
        Reset,
    }

    /// enum for cursor control
    pub enum CursorControl {
        /// move cursor left specified amount of columns
        Left(usize),
        /// move cursor right specified amount of columns
        Right(usize),
        /// move cursor to specified column
        Col(usize),
        /// move cursor specified amount of rows up
        Up(usize),
        /// move cursor specified amount of rows down
        Down(usize),
        /// move cursor to position(col, row)
        Pos(usize, usize),
    }

    /// function to perform an action for display controlling (using a `DisplayControl` enum)
    /// # Arguments
    /// *`sequence`: sequence to perform
    pub fn dc_sequence(sequence: DisplayControl) {
        let seq: String = match sequence {
            DisplayControl::Clear => String::from("\x1b[2J"),
            DisplayControl::ClearLine => String::from("\x1b[2K"),
            DisplayControl::Reset => String::from("\x1bc"),
        };

        print!("{}", seq);
    }
    /// function to perform an action for cursor controlling (using a `CursorControl` enum)
    /// # Arguments
    /// *`sequence`: the sequence to perform
    pub fn cc_sequence(sequence: CursorControl) {
        let seq: String = match sequence {
            CursorControl::Left(n) => format!("\x1b[{}D", n),
            CursorControl::Right(n) => format!("\x1b[{}C", n),
            CursorControl::Col(n) => format!("\x1b[{}G", n),
            CursorControl::Up(n) => format!("\x1b[{}A", n),
            CursorControl::Down(n) => format!("\x1b[{}B", n),
            CursorControl::Pos(x, y) => format!("\x1b[{};{}H", x, y),
        };

        print!("{}", seq);
    }
}

/// terminal for linux
pub struct Terminal {
    stdo: Stdout,
    stdi: Stdin,
    stde: Stderr,
}

impl Terminal {
    // init: {{{

    /// construct a new Terminal from the standard libraries handles
    pub fn new() -> Terminal {
        Terminal {
            stdo: stdout(),
            stdi: stdin(),
            stde: stderr(),
        }
    }

    /// construct a new Terminal from custom handles
    pub fn from_handles(stdo: Stdout, stdi: Stdin, stde: Stderr) -> Terminal {
        Terminal { stdo, stdi, stde }
    }

    // }}}

    // io: {{{

    /// print to stdout with newline
    pub fn println<T: std::fmt::Display>(&self, content: T) -> Result<(), Error> {
        let mut handle = self.stdo.lock();
        match writeln!(handle, "{}", content) {
            Ok(_) => Ok(()),
            Err(_) => Err(Error::new(ErrorKind::Other, "problems writing to stdout")),
        }
    }

    /// print to stdout without newline
    pub fn print<T: std::fmt::Display>(&self, content: T) -> Result<(), Error> {
        let mut handle = self.stdo.lock();
        match write!(handle, "{}", content) {
            Ok(_) => Ok(()),
            Err(_) => Err(Error::new(ErrorKind::Other, "problems writing to stdout")),
        }
    }

    /// print to stderr with newline
    pub fn eprintln<T: std::fmt::Display>(&self, content: T) -> Result<(), Error> {
        let mut handle = self.stde.lock();
        match writeln!(handle, "{}", content) {
            Ok(_) => Ok(()),
            Err(_) => Err(Error::new(ErrorKind::Other, "problems writing to stdout")),
        }
    }

    /// print to stderr without newline
    pub fn eprint<T: std::fmt::Display>(&self, content: T) -> Result<(), Error> {
        let mut handle = self.stde.lock();
        match write!(handle, "{}", content) {
            Ok(_) => Ok(()),
            Err(_) => Err(Error::new(ErrorKind::Other, "problems writing to stdout")),
        }
    }

    /// read a line from stdin and trim any whitespace at the end
    pub fn read_line_trimmed(&self) -> Result<String, Error> {
        let mut input: String = String::new();
        match self.stdi.read_line(&mut input) {
            Ok(_) => (),
            Err(e) => {
                return Err(e);
            }
        }
        Ok(input.trim_end().to_string())
    }

    /// read line from stdin without trimming
    pub fn read_line(&self) -> Result<String, Error> {
        let mut input: String = String::new();
        match self.stdi.read_line(&mut input) {
            Ok(_) => (),
            Err(e) => {
                return Err(e);
            }
        }
        Ok(input)
    }

    // }}}

    // cursor control: {{{
    /// move the cursor the indicated amount in the direction
    /// # Arguments
    /// *`x`: the amuount of columns to move right (can be negative to move left)
    /// *`y`: the amount of rows to go down (can be negative to move up)
    pub fn move_cursor(&self, x: i32, y: i32) -> Result<(), Error> {
        match x.cmp(&0) {
            Ordering::Greater => match self.print(format!("\x1b[{}C", x)) {
                Ok(_) => (),
                Err(e) => {
                    return Err(e);
                }
            },
            Ordering::Less => match self.print(format!("\x1b[{}C", x)) {
                Ok(_) => (),
                Err(e) => {
                    return Err(e);
                }
            },
            Ordering::Equal => (),
        }
        match y.cmp(&0) {
            Ordering::Less => match self.print(format!("\x1b[{}A", y.neg())) {
                Ok(_) => (),
                Err(e) => {
                    return Err(e);
                }
            },
            Ordering::Greater => match self.print(format!("\x1b[{}B", y)) {
                Ok(_) => (),
                Err(e) => {
                    return Err(e);
                }
            },
            Ordering::Equal => (),
        }
        Ok(())
    }

    /// set the cursors position to (x, y)
    /// # Arguments
    /// *`x`: the x position to move to
    /// *`y`: the y position to move to
    pub fn set_cursor(&self, x: usize, y: usize) -> Result<(), Error> {
        self.print(format!("\x1b[{};{}H", x, y))
    }

    /// set the cursors x position
    /// # Arguments
    /// *`x`: the x position to move to
    pub fn set_cursor_x(&self, x: usize) -> Result<(), Error> {
        self.print(format!("\x1b[{}G", x))
    }

    // }}}

    // display control: {{{

    /// clear the current line
    pub fn clear_line(&self) -> Result<(), Error> {
        self.print("\x1b[2K")
    }

    /// clear the screen
    pub fn clear_screen(&self) -> Result<(), Error> {
        self.print("\x1b[2J")
    }

    /// reset the screen
    pub fn reset_screen(&self) -> Result<(), Error> {
        self.print("\x1bc")
    }

    // }}}

    // Widget support: {{{
    pub fn draw_widget<T: widget::Widget>(&self, widget: T) -> Result<(), Error> {
        self.println(widget.render())
    }
    // }}}

}

impl Default for Terminal {
    fn default() -> Self {
        Terminal::new()
    }
}

// vim: fdm=marker
