/// module containing enums to represent terminal effects
pub mod effects {
    /// enum to represent any color
    pub enum Color {
        Black,
        Red,
        Green,
        Yellow,
        Blue,
        Magenta,
        Cyan,
        White,
        /// set color to an rgb color
        RGB(u8, u8, u8),
    }

    /// enum to represent special printings other than colors
    pub enum Special {
        Bold,
        Underline,
        Stroke,
        Italic,
        /// reset all graphical effects
        Reset,
    }
}

/// module for giving printable escape sequences
pub mod print {
    use crate::effects::Color;
    use crate::effects::Special;

    /// change foreground color
    ///
    /// # Arguments
    /// *`col`: color to use
    ///
    /// # Returns
    /// the color code to be printed to the terminal
    ///
    /// # Example
    /// ```rust
    /// use btui::print::fg;
    /// use btui::effects::Color;
    /// println!("{}test", fg(Color::Red));
    /// ```
    pub fn fg(col: Color) -> String {
        match col {
            Color::Black => String::from("\x1b[30m"),
            Color::Red => String::from("\x1b[31m"),
            Color::Green => String::from("\x1b[32m"),
            Color::Yellow => String::from("\x1b[33m"),
            Color::Blue => String::from("\x1b[34m"),
            Color::Magenta => String::from("\x1b[35m"),
            Color::Cyan => String::from("\x1b[36m"),
            Color::White => String::from("\x1b[37m"),
            Color::RGB(r, g, b) => format!("\x1b[38;2;{};{};{}m", r, g, b),
        }
    }

    /// change background color
    ///
    /// # Arguments
    /// *`col`: color to use
    ///
    /// # Returns
    /// the color code to be printed to the terminal
    ///
    /// # Example
    /// ```rust
    /// use btui::print::bg;
    /// use btui::effects::Color;
    /// println!("{}test", bg(Color::Red));
    /// ```
    pub fn bg(col: Color) -> String {
        match col {
            Color::Black => String::from("\x1b[40m"),
            Color::Red => String::from("\x1b[41m"),
            Color::Green => String::from("\x1b[42m"),
            Color::Yellow => String::from("\x1b[43m"),
            Color::Blue => String::from("\x1b[44m"),
            Color::Magenta => String::from("\x1b[45m"),
            Color::Cyan => String::from("\x1b[46m"),
            Color::White => String::from("\x1b[47m"),
            Color::RGB(r, g, b) => format!("\x1b[48;2;{};{};{}m", r, g, b),
        }
    }

    /// apply a special effect
    ///
    /// # Arguments
    /// *`special`: effect to apply
    ///
    /// # Returns
    /// a string containing the escape sequence to use
    ///
    /// # Example
    /// ```rust
    /// use btui::effects::Special;
    /// use btui::print::sp;
    /// println!("{}test{}", sp(Special::Underline), sp(Special::Reset));
    /// ```
    pub fn sp(special: Special) -> String {
        match special {
            Special::Bold => String::from("\x1b[1m"),
            Special::Underline => String::from("\x1b[4m"),
            Special::Stroke => String::from("\x1b[9m"),
            Special::Italic => String::from("\x1b[3m"),
            Special::Reset => String::from("\x1b[0m"),
        }
    }
}
