use super::PartialEnum;

/// Distance between singletons.
#[allow(clippy::len_without_is_empty)]
pub trait Measure<U = Self>: PartialEnum {
	type Len: Default + std::ops::Add<Output = Self::Len> + std::ops::Sub<Output = Self::Len>;

	/// Returns the length of the given element.
	fn len(&self) -> Self::Len;

	/// Returns the distance to the given other element.
	fn distance(&self, other: &U) -> Self::Len;
}

// impl<'a, U, T: Measure<U>> Measure<U> for &'a T {
// 	type Len = T::Len;

// 	fn len(&self) -> Self::Len {
// 		(*self).len()
// 	}
// }

impl Measure for char {
	type Len = u64;

	fn len(&self) -> u64 {
		1
	}

	fn distance(&self, other: &char) -> u64 {
		let a = *self as u64;
		let b = *other as u64;

		if a > b {
			a - b
		} else {
			b - a
		}
	}
}

macro_rules! impl_measure {
	// Measure for type `$ty`.
	// `$cast` is a type that can handle the subtraction of two elements
	// without overflowing.
	// `$len` is a type that can handle the size of the entire domain of `$ty`.
	(@refl $ty:ty, $cast:ty, $len:ty) => {
		impl PartialEnum for $ty {
			const MIN: $ty = <$ty>::MIN;
			const MAX: $ty = <$ty>::MAX;

			fn pred(&self) -> Option<Self> {
				self.checked_sub(1)
			}

			fn succ(&self) -> Option<Self> {
				self.checked_add(1)
			}
		}

		impl_measure!($ty, $ty, $cast, $len);
	};
	(@both $ty1:ty, $ty2:ty, $cast:ty, $len:ty) => {
		impl_measure!($ty1, $ty2, $cast, $len);
		impl_measure!($ty2, $ty1, $cast, $len);
	};
	($ty1:ty, $ty2:ty, $cast:ty, $len:ty) => {
		impl Measure<$ty2> for $ty1 {
			type Len = $len;

			fn len(&self) -> $len {
				1
			}

			fn distance(&self, other: &$ty2) -> $len {
				let a = *self as $cast;
				let b = *other as $cast;

				if a > b {
					(a - b) as $len
				} else {
					(b - a) as $len
				}
			}
		}
	};
}

// All of those are generated by the `generate-measures.rb` script
// to avoid mistakes.
impl_measure!(@refl u8, u8, u16);
impl_measure!(@refl u16, u16, u32);
impl_measure!(@refl u32, u32, u64);
impl_measure!(@refl u64, u64, u128);
impl_measure!(@refl i8, i16, u8);
impl_measure!(@refl i16, i32, u16);
impl_measure!(@refl i32, i64, u32);
impl_measure!(@refl i64, i128, u64);
impl_measure!(@both u8, u16, u16, u32);
impl_measure!(@both u8, u32, u32, u64);
impl_measure!(@both u8, u64, u64, u128);
impl_measure!(@both u8, i8, i16, u16);
impl_measure!(@both u8, i16, i32, u16);
impl_measure!(@both u8, i32, i64, u32);
impl_measure!(@both u8, i64, i128, u64);
impl_measure!(@both u16, u32, u32, u64);
impl_measure!(@both u16, u64, u64, u128);
impl_measure!(@both u16, i8, i32, u32);
impl_measure!(@both u16, i16, i32, u32);
impl_measure!(@both u16, i32, i64, u32);
impl_measure!(@both u16, i64, i128, u64);
impl_measure!(@both u32, u64, u64, u128);
impl_measure!(@both u32, i8, i64, u64);
impl_measure!(@both u32, i16, i64, u64);
impl_measure!(@both u32, i32, i64, u64);
impl_measure!(@both u32, i64, i128, u64);
impl_measure!(@both u64, i8, i128, u128);
impl_measure!(@both u64, i16, i128, u128);
impl_measure!(@both u64, i32, i128, u128);
impl_measure!(@both u64, i64, i128, u128);
impl_measure!(@both i8, i16, i32, u16);
impl_measure!(@both i8, i32, i64, u32);
impl_measure!(@both i8, i64, i128, u64);
impl_measure!(@both i16, i32, i64, u32);
impl_measure!(@both i16, i64, i128, u64);
impl_measure!(@both i32, i64, i128, u64);
#[cfg(target_pointer_width = "8")]
impl_measure!(@refl usize, u8, u16);
#[cfg(target_pointer_width = "8")]
impl_measure!(@both usize, u8, u8, u16);
#[cfg(target_pointer_width = "8")]
impl_measure!(@both usize, u16, u16, u32);
#[cfg(target_pointer_width = "8")]
impl_measure!(@both usize, u32, u32, u64);
#[cfg(target_pointer_width = "8")]
impl_measure!(@both usize, u64, u64, u128);
#[cfg(target_pointer_width = "8")]
impl_measure!(@both usize, i8, i16, u16);
#[cfg(target_pointer_width = "8")]
impl_measure!(@both usize, i16, i32, u16);
#[cfg(target_pointer_width = "8")]
impl_measure!(@both usize, i32, i64, u32);
#[cfg(target_pointer_width = "8")]
impl_measure!(@both usize, i64, i128, u64);
#[cfg(target_pointer_width = "16")]
impl_measure!(@refl usize, u16, u32);
#[cfg(target_pointer_width = "16")]
impl_measure!(@both usize, u8, u16, u32);
#[cfg(target_pointer_width = "16")]
impl_measure!(@both usize, u16, u16, u32);
#[cfg(target_pointer_width = "16")]
impl_measure!(@both usize, u32, u32, u64);
#[cfg(target_pointer_width = "16")]
impl_measure!(@both usize, u64, u64, u128);
#[cfg(target_pointer_width = "16")]
impl_measure!(@both usize, i8, i32, u32);
#[cfg(target_pointer_width = "16")]
impl_measure!(@both usize, i16, i32, u32);
#[cfg(target_pointer_width = "16")]
impl_measure!(@both usize, i32, i64, u32);
#[cfg(target_pointer_width = "16")]
impl_measure!(@both usize, i64, i128, u64);
#[cfg(target_pointer_width = "32")]
impl_measure!(@refl usize, u32, u64);
#[cfg(target_pointer_width = "32")]
impl_measure!(@both usize, u8, u32, u64);
#[cfg(target_pointer_width = "32")]
impl_measure!(@both usize, u16, u32, u64);
#[cfg(target_pointer_width = "32")]
impl_measure!(@both usize, u32, u32, u64);
#[cfg(target_pointer_width = "32")]
impl_measure!(@both usize, u64, u64, u128);
#[cfg(target_pointer_width = "32")]
impl_measure!(@both usize, i8, i64, u64);
#[cfg(target_pointer_width = "32")]
impl_measure!(@both usize, i16, i64, u64);
#[cfg(target_pointer_width = "32")]
impl_measure!(@both usize, i32, i64, u64);
#[cfg(target_pointer_width = "32")]
impl_measure!(@both usize, i64, i128, u64);
#[cfg(target_pointer_width = "64")]
impl_measure!(@refl usize, u64, u128);
#[cfg(target_pointer_width = "64")]
impl_measure!(@both usize, u8, u64, u128);
#[cfg(target_pointer_width = "64")]
impl_measure!(@both usize, u16, u64, u128);
#[cfg(target_pointer_width = "64")]
impl_measure!(@both usize, u32, u64, u128);
#[cfg(target_pointer_width = "64")]
impl_measure!(@both usize, u64, u64, u128);
#[cfg(target_pointer_width = "64")]
impl_measure!(@both usize, i8, i128, u128);
#[cfg(target_pointer_width = "64")]
impl_measure!(@both usize, i16, i128, u128);
#[cfg(target_pointer_width = "64")]
impl_measure!(@both usize, i32, i128, u128);
#[cfg(target_pointer_width = "64")]
impl_measure!(@both usize, i64, i128, u128);

macro_rules! impl_f_measure {
	(@refl $ty:ty, $len:ty) => {
		impl PartialEnum for $ty {
			const MIN: $ty = <$ty>::NEG_INFINITY;
			const MAX: $ty = <$ty>::INFINITY;

			fn pred(&self) -> Option<Self> {
				None
			}

			fn succ(&self) -> Option<Self> {
				None
			}
		}

		impl_f_measure!($ty, $ty, $ty, $len);
	};
	(@both $ty1:ty, $ty2:ty, $cast:ty, $len:ty) => {
		impl_f_measure!($ty1, $ty2, $cast, $len);
		impl_f_measure!($ty2, $ty1, $cast, $len);
	};
	($ty1:ty, $ty2:ty, $cast:ty, $len:ty) => {
		impl Measure<$ty2> for $ty1 {
			type Len = $len;

			fn len(&self) -> $len {
				0.0
			}

			fn distance(&self, other: &$ty2) -> $len {
				if self.is_infinite() || other.is_infinite() {
					Self::INFINITY
				} else {
					let a = *self as $cast;
					let b = *other as $cast;

					if a > b {
						(a - b) as $len
					} else {
						(b - a) as $len
					}
				}
			}
		}
	};
}

impl_f_measure!(@refl f32, f32);
impl_f_measure!(@refl f64, f64);
