use crate::{bsc_wrapper::BscChainApi, error::CustomErrors};
use serde::Deserialize;
use serde_json;
use web3::types::{H160, H256, U256, U64};

use super::Response;

pub struct Transactions;

impl Response<Vec<Tx>> {
    pub async fn parse_str(response: String) -> Response<Vec<Tx>> {
        serde_json::from_str::<Response<Vec<Tx>>>(&response).unwrap()
    }
}

#[derive(Debug, Deserialize, Clone)]
pub struct Tx {
    #[serde(rename = "blockNumber")]
    block_no: String,
    #[serde(rename = "timeStamp")]
    timestamp: String,
    #[serde(default)]
    hash: String,
    #[serde(default)]
    nonce: String,
    #[serde(default)]
    #[serde(rename = "blockHash")]
    block_hash: String,
    #[serde(default)]
    #[serde(rename = "transactionIndex")]
    transaction_index: String,
    from: H160,
    to: H160,
    value: String,
    gas: String,
    #[serde(default)]
    #[serde(rename = "gasPrice")]
    gas_price: String,
    #[serde(rename = "isError")]
    is_error: String,
    #[serde(default)]
    txreceipt_status: String,
    input: String,
    #[serde(default)]
    #[serde(rename = "contractAddress")]
    contract_address: String,
    #[serde(default)]
    #[serde(rename = "cumulativeGasUsed")]
    cumulative_gas_used: String,
    #[serde(rename = "gasUsed")]
    gas_used: String,
    #[serde(default)]
    confirmations: String,
    #[serde(default)]
    #[serde(rename = "traceId")]
    trace_id: String,
    #[serde(default)]
    #[serde(rename = "errCode")]
    err_code: String,
    #[serde(default)]
    #[serde(rename = "type")]
    _type: String,
}

impl Tx {
    pub fn block_no(&self) -> U64 {
        U64::from_dec_str(&self.block_no).unwrap_or_default()
    }

    pub fn timestamp(&self) -> U256 {
        U256::from_dec_str(&self.block_no).unwrap_or_default()
    }

    pub fn hash(&self) -> Option<H256> {
        if self.hash.is_empty() {
            return None;
        } else {
            let mut slice = vec![];
            U256::from_dec_str(&self.hash)
                .unwrap_or_default()
                .to_big_endian(slice.as_mut());
            let hash = H256::from_slice(&slice);
            Some(hash)
        }
    }

    pub fn nonce(&self) -> Option<U256> {
        if self.nonce.is_empty() {
            None
        } else {
            Some(U256::from_dec_str(&self.block_no).unwrap_or_default())
        }
    }

    pub fn block_hash(&self) -> Option<H256> {
        if self.block_hash.is_empty() {
            return None;
        } else {
            let mut slice = vec![];
            U256::from_dec_str(&self.block_hash)
                .unwrap_or_default()
                .to_big_endian(slice.as_mut());
            let block_hash = H256::from_slice(&slice);
            Some(block_hash)
        }
    }
    pub fn transaction_index(&self) -> Option<u16> {
        if self.transaction_index.is_empty() {
            None
        } else {
            Some(self.transaction_index.parse::<u16>().unwrap_or(0))
        }
    }

    pub fn from(&self) -> H160 {
        self.from
    }

    pub fn to(&self) -> H160 {
        self.to
    }

    pub fn value(&self) -> Option<U256> {
        if self.value.is_empty() {
            None
        } else {
            Some(U256::from_dec_str(&self.value).unwrap_or_default())
        }
    }

    pub fn gas(&self) -> U256 {
        U256::from_dec_str(&self.gas).unwrap_or_default()
    }

    pub fn gas_price(&self) -> Option<U256> {
        if self.gas_price.is_empty() {
            None
        } else {
            Some(U256::from_dec_str(&self.gas_price).unwrap_or_default())
        }
    }

    pub fn gas_used(&self) -> U256 {
        U256::from_dec_str(&self.gas_used).unwrap_or_default()
    }

    pub fn is_error(&self) -> String {
        self.is_error.clone()
    }

    pub fn txreceipt_status(&self) -> Option<String> {
        if self.txreceipt_status.is_empty() {
            None
        } else {
            Some(self.txreceipt_status.clone())
        }
    }

    pub fn input(&self) -> String {
        self.input.clone()
    }

    pub fn contract_address(&self) -> Option<H160> {
        if self.contract_address.is_empty() {
            None
        } else {
            let contract_address: H160 = self.contract_address.parse().unwrap();
            Some(contract_address)
        }
    }

    pub fn cumulative_gas_used(&self) -> Option<U256> {
        if self.cumulative_gas_used.is_empty() {
            None
        } else {
            Some(U256::from_dec_str(&self.cumulative_gas_used).unwrap_or_default())
        }
    }

    pub fn confirmations(&self) -> Option<usize> {
        if self.confirmations.is_empty() {
            None
        } else {
            Some(self.confirmations.parse::<usize>().unwrap_or(0))
        }
    }

    pub fn trace_id(&self) -> Option<String> {
        if self.trace_id.is_empty() {
            None
        } else {
            Some(self.trace_id.clone())
        }
    }

    pub fn err_code(&self) -> Option<String> {
        if self.err_code.is_empty() {
            None
        } else {
            Some(self.err_code.clone())
        }
    }

    pub fn _type(&self) -> Option<String> {
        if self._type.is_empty() {
            None
        } else {
            Some(self._type.clone())
        }
    }
}

#[derive(Debug, Deserialize, Clone)]
struct Status {
    status: String,
}

impl Response<Status> {
    pub async fn parse_str(response: String) -> Response<Status> {
        serde_json::from_str::<Response<Status>>(&response).unwrap()
    }

    pub fn transaction_status(self) -> Result<bool, CustomErrors> {
        Ok(self.result()?.status == "1")
    }
}

impl Transactions {
    pub async fn transactions(
        &self,
        api: &mut BscChainApi,
        address: &str,
        start_block: u32,
        end_block: u32,
        asc_sort: bool,
    ) -> Result<Vec<Tx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "txlist");
        api.query.add_params("address", address);
        api.query.add_params("startblock", &start_block.to_string());
        api.query.add_params("endblock", &end_block.to_string());
        api.query
            .add_params("sort", if asc_sort { "asc" } else { "dsc" });

        Ok(Response::<Vec<Tx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    // pub async fn transactions_pagination(
    //     &self,
    //     api: &mut BscChainApi,
    //     address: &str,
    //     asc_sort: bool,
    //     page: u16,
    //     offset: u32,
    // ) -> Result<Vec<Tx>, CustomErrors> {
    //     api.query.add_params("apikey", &api.api_key);
    //     api.query.add_params("module", "account");
    //     api.query.add_params("action", "txlist");
    //     api.query.add_params("address", address);
    //     api.query.add_params("page", &page.to_string());
    //     api.query.add_params("offset", &offset.to_string());
    //     api.query.add_params("startblock", "0");
    //     api.query.add_params("endblock", "latest");
    //     api.query
    //         .add_params("sort", if asc_sort { "asc" } else { "dsc" });

    //     let resp = api
    //         .client
    //         .get(&api.query.build_url())
    //         .send()
    //         .await?
    //         .text()
    //         .await?;

    //     dbg!(resp);

    //     Ok(Response::<Vec<Tx>>::parse_str(
    //         api.client
    //             .get(&api.query.build_url())
    //             .send()
    //             .await?
    //             .text()
    //             .await?,
    //     )
    //     .await
    //     .result())
    // }

    pub async fn internal_transactions(
        &self,
        api: &mut BscChainApi,
        address: &str,
        asc_sort: bool,
    ) -> Result<Vec<Tx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "txlistinternal");
        api.query.add_params("address", address);
        api.query.add_params("startblock", "0");
        api.query.add_params("endblock", "latest");
        api.query
            .add_params("sort", if asc_sort { "asc" } else { "dsc" });
        Ok(Response::<Vec<Tx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    pub async fn internal_transaction_pagination(
        &self,
        api: &mut BscChainApi,
        address: &str,
        asc_sort: bool,
        page: u16,
        offset: u32,
    ) -> Result<Vec<Tx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "txlistinternal");
        api.query.add_params("address", address);
        api.query.add_params("startblock", "0");
        api.query.add_params("endblock", "latest");
        api.query
            .add_params("sort", if asc_sort { "asc" } else { "dsc" });
        api.query.add_params("page", &page.to_string());
        api.query.add_params("offset", &offset.to_string());

        Ok(Response::<Vec<Tx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    pub async fn get_internal_transaction_by_tx(
        &self,
        api: &mut BscChainApi,
        tx: &str,
    ) -> Result<Vec<Tx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "txlistinternal");
        api.query.add_params("txhash", tx);

        Ok(Response::<Vec<Tx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    pub async fn general_internal_transaction(
        &self,
        api: &mut BscChainApi,
        start_block: u32,
        end_block: u32,
        asc_sort: bool,
        page: u16,
        offset: u32,
    ) -> Result<Vec<Tx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "txlistinternal");
        api.query.add_params("startblock", &start_block.to_string());
        api.query.add_params("endblock", &end_block.to_string());
        api.query
            .add_params("sort", if asc_sort { "asc" } else { "dsc" });
        api.query.add_params("page", &page.to_string());
        api.query.add_params("offset", &offset.to_string());

        Ok(Response::<Vec<Tx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    pub async fn check_tx_status(
        &self,
        api: &mut BscChainApi,
        txhash: &str,
    ) -> Result<bool, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "transaction");
        api.query.add_params("action", "gettxreceiptstatus");
        api.query.add_params("txhash", txhash);

        Ok(Response::<Status>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .transaction_status()?)
    }
}

#[cfg(test)]
mod test {
    use super::*;
    use actix_rt;

    fn create_success() -> BscChainApi {
        BscChainApi::new("YOUR_API_KEY_HERE")
    }

    // #[actix_rt::test]
    // async fn transaction_pagination() {
    //     let mut api = create_success();
    //     let m = Transactions
    //         .transactions_pagination(&mut api,"0x0000000000000000000000000000000000001004", true, 20, 6)
    //         .await;
    //     dbg!(&m);
    //     assert_eq!(
    //         m.as_ref().unwrap()[2].to(),
    //         H160::from_slice(&hex_literal::hex!(
    //             "0000000000000000000000000000000000001004"
    //         ))
    //     );
    // }

    #[actix_rt::test]
    async fn internal_transactions() {
        let mut api = create_success();
        let m = Transactions
            .internal_transactions(&mut api, "0x0000000000000000000000000000000000001004", true)
            .await;
        dbg!(&m);

        assert_eq!(
            m.as_ref().unwrap()[2].from(),
            H160::from_slice(&hex_literal::hex!(
                "0000000000000000000000000000000000001004"
            ))
        );
    }

    #[actix_rt::test]
    async fn internal_transaction_pagination() {
        let mut api = create_success();
        let m = Transactions
            .internal_transaction_pagination(
                &mut api,
                "0x0000000000000000000000000000000000001004",
                true,
                1,
                10,
            )
            .await;
        dbg!(&m);

        assert_eq!(
            m.as_ref().unwrap()[2].from(),
            H160::from_slice(&hex_literal::hex!(
                "0000000000000000000000000000000000001004"
            ))
        );
    }

    #[actix_rt::test]
    async fn internal_transaction_general_pagination() {
        let mut api = create_success();
        let m = Transactions
            .general_internal_transaction(&mut api, 0, 9999999, true, 1, 10)
            .await;
        dbg!(&m);
        assert_eq!(
            m.as_ref().unwrap()[2].from(),
            H160::from_slice(&hex_literal::hex!(
                "0000000000000000000000000000000000001004"
            ))
        );
    }

    #[actix_rt::test]
    async fn internal_transaction_tx() {
        let mut api = create_success();
        let m = Transactions
            .get_internal_transaction_by_tx(
                &mut api,
                "0x4d74a6fc84d57f18b8e1dfa07ee517c4feb296d16a8353ee41adc03669982028",
            )
            .await;
        dbg!(&m);

        assert_eq!(
            m.as_ref().unwrap()[0].from(),
            H160::from_slice(&hex_literal::hex!(
                "0000000000000000000000000000000000001004"
            ))
        );
    }

    #[actix_rt::test]
    async fn transaction_status() {
        let mut api = create_success();
        let m = Transactions
            .check_tx_status(
                &mut api,
                "0xe9975702518c79caf81d5da65dea689dcac701fcdd063f848d4f03c85392fd00",
            )
            .await;
        assert_eq!(m.unwrap(), true);
    }
}
