use web3::types::U256;

use crate::{error::CustomErrors};

use super::{BscChainApi, Response};

pub struct Token;

impl Token {
    pub async fn bep_20_get_total_supply(
        &self,
        api: &mut BscChainApi,
        address: &str,
    ) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "stats");
        api.query.add_params("action", "tokensupply");
        api.query.add_params("contractaddress", address);

        Ok(Response::<U256>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?)
    }

    pub async fn bep_20_get_circulating_supply(
        &self,
        api: &mut BscChainApi,
        address: &str,
    ) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "stats");
        api.query.add_params("action", "tokenCsupply");
        api.query.add_params("contractaddress", address);

        Ok(Response::<U256>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?)
    }

    pub async fn token_balance(
        &self,
        api: &mut BscChainApi,
        address: &str,
        token_contract_address: &str,
    ) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "tokenbalance");
        api.query.add_params("tag", "latest");
        api.query.add_params("address", address);
        api.query
            .add_params("contractaddress", token_contract_address);

        Ok(Response::<U256>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?)
    }
}

#[cfg(test)]
mod test {
    use super::*;

    fn create_success() -> BscChainApi {
        BscChainApi::new("YOUR_API_KEY_HERE")
    }
    #[actix_rt::test]
    async fn get_tot_supply() {
        let mut m = create_success();
        let s = Token
            .bep_20_get_total_supply(&mut m, "0xe9e7cea3dedca5984780bafc599bd69add087d56")
            .await
            .unwrap();
        assert_eq!(s, U256::from(4200999999996203280118545633u128));
    }

    #[actix_rt::test]
    async fn get_tot_c_supply() {
        let mut m = create_success();
        let s = Token
            .bep_20_get_circulating_supply(&mut m, "0xe9e7cea3dedca5984780bafc599bd69add087d56")
            .await
            .unwrap();
        assert_eq!(s, U256::from(4090363228959545470118545633u128));
    }

    // For Testing Purpose twt token is used
    #[actix_rt::test]
    async fn test_token_balance() {
        let mut api = create_success();
        let m = Token
            .token_balance(&mut api,
                "0x0000000000000000000000000000000000001004",
                "0x4b0f1812e5df2a09796481ff14017e6005508003",
            )
            .await;
        assert_eq!(m.unwrap(), U256::from(816370091928590930000000000u128))
    }

}
