use serde::Deserialize;
use web3::types::{H160, U256};

use crate::{error::CustomErrors, BscChainApi};

use super::Response;

pub struct Stats;

#[derive(Debug, Deserialize, Clone)]
pub struct Bnb {
    ethbtc: String,
    ethbtc_timestamp: String,
    ethusd: String,
    ethusd_timestamp: String,
}

impl Bnb {
    pub fn ethbtc(&self) -> f64 {
        self.ethbtc.parse().unwrap()
    }

    pub fn ethusd(&self) -> f64 {
        self.ethusd.parse().unwrap()
    }

    pub fn ethbtc_timestamp(&self) -> U256 {
        self.ethbtc_timestamp.parse().unwrap()
    }

    pub fn ethusd_timestamp(&self) -> U256 {
        self.ethusd_timestamp.parse().unwrap()
    }
}

#[derive(Debug, Deserialize, Clone)]
pub struct Validator {
    #[serde(rename = "validatorAddress")]
    validator_address: H160,
    #[serde(rename = "validatorName")]
    validator_name: String,
    #[serde(rename = "validatorStatus")]
    validator_status: String,
    #[serde(rename = "validatorVotingPower")]
    validator_voting_power: String,
    #[serde(rename = "validatorVotingPowerProportion")]
    validator_voting_power_proportion: String,
}

impl Validator {
    pub fn validator_address(&self) -> H160 {
        self.validator_address
    }

    pub fn validator_name(&self) -> String {
        self.validator_name.clone()
    }

    pub fn validator_status(&self) -> String {
        self.validator_status.clone()
    }

    pub fn validator_voting_power(&self) -> U256 {
        U256::from_dec_str(&self.validator_voting_power).unwrap()
    }

    pub fn validator_voting_power_proportion(&self) -> f64 {
        self.validator_voting_power_proportion.parse().unwrap()
    }
}

impl Response<Vec<Validator>> {
    pub async fn parse_str(response: String) -> Response<Vec<Validator>> {
        serde_json::from_str::<Response<Vec<Validator>>>(&response).unwrap()
    }
}

impl Response<Bnb> {
    pub async fn parse_str(response: String) -> Response<Bnb> {
        serde_json::from_str::<Response<Bnb>>(&response).unwrap()
    }
}

impl Stats {
    pub async fn get_bnb_total_supply(&self, api: &mut BscChainApi) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "stats");
        api.query.add_params("action", "bnbsupply");

        Ok(Response::<U256>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?)
    }

    pub async fn get_validator_list(
        &self,
        api: &mut BscChainApi,
    ) -> Result<Vec<Validator>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "stats");
        api.query.add_params("action", "validators");

        Ok(Response::<Vec<Validator>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    pub async fn get_bnb_price(&self, api: &mut BscChainApi) -> Result<Bnb, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "stats");
        api.query.add_params("action", "bnbprice");

        Ok(Response::<Bnb>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }
}

#[cfg(test)]
mod test {
    use super::*;

    fn create_success() -> BscChainApi {
        BscChainApi::new("YOUR_API_KEY_HERE")
    }
    #[actix_rt::test]
    async fn get_tot_supply() {
        let mut m = create_success();
        let s = Stats.get_bnb_total_supply(&mut m).await.unwrap();
        assert_eq!(s, U256::from(15651839922666400000000000u128));
    }

    #[actix_rt::test]
    async fn get_validators() {
        let mut m = create_success();
        let s = Stats.get_validator_list(&mut m).await.unwrap();
        dbg!(&s);
        assert_eq!(s[0].validator_voting_power_proportion(), 0.0617);
    }

    #[actix_rt::test]
    async fn bnb_price() {
        let mut m = create_success();
        let s = Stats.get_bnb_price(&mut m).await.unwrap();
        dbg!(&s);
        assert!(true)
    }
}
