use web3::types::U256;
use serde::Deserialize;
use web3::types::{Block, Bytes, Transaction, TransactionReceipt};

use super::BscChainApi;
use crate::error::{CustomErrors, ErrorCause};

pub struct Proxy;

#[derive(Debug, Deserialize)]
pub struct ErrorDetails {
    code: i64,
    message: String,
}
#[derive(Debug, Deserialize)]
pub struct Error {
    error: ErrorDetails,
}

#[derive(Debug, Deserialize)]
pub struct EthResponse<T> {
    jsonrpc: String,
    id: u16,
    result: T,
}

impl<T> EthResponse<T> {
    pub fn jsonrpc(&self) -> f32 {
        self.jsonrpc.parse::<f32>().unwrap()
    }

    pub fn id(&self) -> u16 {
        self.id
    }

    pub fn result(self) -> T {
        self.result
    }
}

impl EthResponse<U256> {
    pub async fn parse_str(response: String) -> Result<EthResponse<U256>, CustomErrors> {
        if response.contains("error") {
            let err = serde_json::from_str::<Error>(&response).unwrap();
            Err(CustomErrors::new(ErrorCause::BadRequest(err.error.message)))
        } else {
            Ok(serde_json::from_str::<EthResponse<U256>>(&response).unwrap())
        }
    }
}

impl EthResponse<Transaction> {
    pub async fn parse_str(response: String) -> Result<EthResponse<Transaction>, CustomErrors> {
        if response.contains("error") {
            let err = serde_json::from_str::<Error>(&response).unwrap();
            Err(CustomErrors::new(ErrorCause::BadRequest(err.error.message)))
        } else {
            Ok(serde_json::from_str::<EthResponse<Transaction>>(&response).unwrap())
        }
    }
}

impl EthResponse<TransactionReceipt> {
    pub async fn parse_str(
        response: String,
    ) -> Result<EthResponse<TransactionReceipt>, CustomErrors> {
        if response.contains("error") {
            let err = serde_json::from_str::<Error>(&response).unwrap();
            Err(CustomErrors::new(ErrorCause::BadRequest(err.error.message)))
        } else {
            Ok(serde_json::from_str::<EthResponse<TransactionReceipt>>(&response).unwrap())
        }
    }
}

impl EthResponse<Block<Transaction>> {
    pub async fn parse_str(
        response: String,
    ) -> Result<EthResponse<Block<Transaction>>, CustomErrors> {
        if response.contains("error") {
            let err = serde_json::from_str::<Error>(&response).unwrap();
            Err(CustomErrors::new(ErrorCause::BadRequest(err.error.message)))
        } else {
            Ok(serde_json::from_str::<EthResponse<Block<Transaction>>>(&response).unwrap())
        }
    }
}

impl EthResponse<Bytes> {
    pub async fn parse_str(response: String) -> Result<EthResponse<Bytes>, CustomErrors> {
        if response.contains("error") {
            let err = serde_json::from_str::<Error>(&response).unwrap();
            Err(CustomErrors::new(ErrorCause::BadRequest(err.error.message)))
        } else {
            Ok(serde_json::from_str::<EthResponse<Bytes>>(&response).unwrap())
        }
    }
}

impl Proxy {
    pub async fn block_number(&self, api: &mut BscChainApi) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "proxy");
        api.query.add_params("action", "eth_blockNumber");

        Ok(EthResponse::<U256>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?
        .result())
    }

    pub async fn get_blocks_by_number(
        &self,
        api: &mut BscChainApi,
        block_no: u64,
    ) -> Result<Block<Transaction>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "proxy");
        api.query.add_params("action", "eth_getBlockByNumber");
        api.query.add_params("tag", &format!("0x{:x}", block_no));
        api.query.add_params("boolean", "true");

        Ok(EthResponse::<Block<Transaction>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?
        .result())
    }

    pub async fn get_block_tx_count(
        &self,
        api: &mut BscChainApi,
        block_no: u32,
    ) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "proxy");
        api.query
            .add_params("action", "eth_getBlockTransactionCountByNumber");
        api.query.add_params("tag", &format!("0x{:x}", block_no));

        Ok(EthResponse::<U256>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?
        .result())
    }

    pub async fn get_transaction_count(
        &self,
        api: &mut BscChainApi,
        address: &str,
    ) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "proxy");
        api.query.add_params("action", "eth_getTransactionCount");
        api.query.add_params("address", address);
        api.query.add_params("tag", "latest");

        Ok(EthResponse::<U256>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?
        .result())
    }

    pub async fn get_gas_price(&self, api: &mut BscChainApi) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "proxy");
        api.query.add_params("action", "eth_gasPrice");

        Ok(EthResponse::<U256>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?
        .result())
    }

    pub async fn get_transaction_by_hash(
        &self,
        api: &mut BscChainApi,
        tx_hash: &str,
    ) -> Result<Transaction, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "proxy");
        api.query.add_params("action", "eth_getTransactionByHash");
        api.query.add_params("txhash", tx_hash);

        Ok(EthResponse::<Transaction>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?
        .result())
    }

    pub async fn get_transaction_recipt(
        &self,
        api: &mut BscChainApi,
        tx_hash: &str,
    ) -> Result<TransactionReceipt, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "proxy");
        api.query.add_params("action", "eth_getTransactionReceipt");
        api.query.add_params("txhash", tx_hash);

        Ok(EthResponse::<TransactionReceipt>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?
        .result())
    }

    pub async fn get_transaction_by_block_no_and_index(
        &self,
        api: &mut BscChainApi,
        block_no: u64,
        index: u16,
    ) -> Result<Transaction, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "proxy");
        api.query
            .add_params("action", "eth_getTransactionByBlockNumberAndIndex");
        api.query.add_params("tag", &format!("0x{:x}", block_no));
        api.query.add_params("index", &format!("0x{:x}", index));
        Ok(EthResponse::<Transaction>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?
        .result())
    }

    pub async fn get_code(
        &self,
        api: &mut BscChainApi,
        address: &str,
    ) -> Result<Bytes, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "proxy");
        api.query.add_params("action", "eth_getCode");
        api.query.add_params("address", address);
        api.query.add_params("tag", "latest");

        Ok(EthResponse::<Bytes>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?
        .result())
    }
}

#[cfg(test)]
mod test {
    use std::str::FromStr;

    use web3::types::{H160, U64};

    use super::*;

    fn create_success() -> BscChainApi {
        BscChainApi::new("YOUR_API_KEY_HERE")
    }

    #[actix_rt::test]
    async fn eth_block_number() {
        let mut m = create_success();
        Proxy.block_number(&mut m).await.unwrap();
        assert!(true);
    }

    #[actix_rt::test]
    async fn eth_tx_count() {
        let mut m = create_success();
        assert_eq!(
            Proxy.get_block_tx_count(&mut m, 7373770).await.unwrap(),
            U256::from(413)
        );
    }

    #[actix_rt::test]
    async fn gas_price() {
        let mut m = create_success();
        let i = Proxy.get_gas_price(&mut m).await.unwrap();
        dbg!(i);

        assert_eq!(i, U256::from(5000000000u128))
    }

    #[actix_rt::test]
    async fn tx_count() {
        let mut m = create_success();
        let i = Proxy
            .get_transaction_count(&mut m, "0xdB13FCA9C10805c39683FF8b2642648C573d1437")
            .await
            .unwrap();
        dbg!(i);

        assert_eq!(i, U256::from(106))
    }

    #[actix_rt::test]
    async fn get_by_tx() {
        let mut m = create_success();
        let i = Proxy
            .get_transaction_by_hash(
                &mut m,
                "0x74e57cf3c2c4413c9bd02c43ae4d5f3e05ccc77b7cba7a83314e0d03d5fab8db",
            )
            .await
            .unwrap();
        dbg!(&i);

        assert_eq!(i.block_number.unwrap(), U64::from(7490815))
    }

    #[actix_rt::test]
    async fn get_block_by_num() {
        let mut m = create_success();
        let i = Proxy.get_blocks_by_number(&mut m, 4000000).await.unwrap();
        dbg!(&i);

        assert_eq!(
            i.author,
            H160::from_str("4430b3230294d12c6ab2aac5c2cd68e80b16b581").unwrap()
        )
    }

    #[actix_rt::test]
    async fn get_tx_by_b_and_i() {
        let mut m = create_success();
        let i = Proxy
            .get_transaction_by_block_no_and_index(&mut m, 4000000, 0)
            .await
            .unwrap();
        dbg!(&i);

        assert_eq!(i.block_number.unwrap(), U64::from(4000000))
    }

    #[actix_rt::test]
    async fn get_code() {
        let mut m = create_success();
        let i = Proxy
            .get_code(&mut m, "0x05fF2B0DB69458A0750badebc4f9e13aDd608C7F")
            .await
            .unwrap();
        dbg!(i);

        assert!(true)
    }
}
