pub mod accounts;
pub mod blocks;
pub mod contract;
pub mod proxy;
pub mod token;
pub mod transactions;
pub mod stats;

use crate::{
    error::{CustomErrors, ErrorCause},
    query_handler::QueryBuilder,
};
use reqwest::Client;
use serde::Deserialize;

#[derive(Debug, Deserialize)]
pub struct Response<T: Clone> {
    status: String,
    message: String,
    result: T,
}

impl<T> Response<T>
where
    T: Clone,
{
    pub fn new(status: String, message: String, result: T) -> Response<T> {
        Response {
            status,
            message,
            result,
        }
    }

    pub fn status(&self) -> String {
        self.status.clone()
    }

    pub fn message(&self) -> String {
        self.message.clone()
    }

    pub fn result(self) -> Result<T, CustomErrors> {
        if self.status() != "1" {
            return Err(CustomErrors::new(ErrorCause::BadRequest(self.message())));
        } else {
            Ok(self.result)
        }
    }
}

pub struct BscChainApi {
    api_key: String,
    query: QueryBuilder,
    client: Client,
}

impl BscChainApi {
    pub fn new(api_key: &str) -> BscChainApi {
        BscChainApi {
            api_key: api_key.to_string(),
            query: QueryBuilder::new("api.bscscan.com", "api"),
            client: Client::builder().https_only(true).build().unwrap(),
        }
    }
}
