use super::{BscChainApi, Response};
use crate::{error::CustomErrors};
use serde::Deserialize;

pub struct ContractApi;

#[derive(Debug, Deserialize, Clone)]
pub struct Code {
    #[serde(rename = "SourceCode")]
    sourcecode: String,
    #[serde(rename = "ContractName")]
    contract_name: String,
    #[serde(rename = "CompilerVersion")]
    compile_version: String,
    #[serde(rename = "OptimizationUsed")]
    optimization_used: String,
    #[serde(rename = "Runs")]
    runs: String,
    #[serde(rename = "ConstructorArguments")]
    constructor_arguments: String,
    #[serde(rename = "EVMVersion")]
    evm_version: String,
    #[serde(rename = "Library")]
    library: String,
    #[serde(rename = "LicenseType")]
    license_type: String,
    #[serde(rename = "Proxy")]
    proxy: String,
    #[serde(rename = "Implementation")]
    implementation: String,
    #[serde(rename = "SwarmSource")]
    swarm_source: String,
}

impl Code {
    pub fn sourcecode(&self) -> String {
        self.sourcecode.clone()
    }

    pub fn contract_name(&self) -> String {
        self.contract_name.clone()
    }

    pub fn compile_version(&self) -> String {
        self.compile_version.clone()
    }

    pub fn optimization_used(&self) -> String {
        self.optimization_used.clone()
    }

    pub fn runs(&self) -> String {
        self.runs.clone()
    }

    pub fn constructor_arguments(&self) -> String {
        self.constructor_arguments.clone()
    }

    pub fn evm_version(&self) -> String {
        self.evm_version.clone()
    }

    pub fn library(&self) -> String {
        self.library.clone()
    }

    pub fn license_type(&self) -> String {
        self.license_type.clone()
    }

    pub fn proxy(&self) -> String {
        self.proxy.clone()
    }

    pub fn implementation(&self) -> String {
        self.implementation.clone()
    }

    pub fn swarm_source(&self) -> String {
        self.swarm_source.clone()
    }
}

impl Response<Vec<Code>> {
    pub async fn parse_str(response: String) -> Response<Vec<Code>> {
        serde_json::from_str::<Response<Vec<Code>>>(&response).unwrap()
    }
}

impl Response<String> {
    pub async fn parse_str(response: String) -> Result<String, CustomErrors> {
        let m = serde_json::from_str::<Response<String>>(&response).unwrap();
        Ok(m.result()?)
    }
}

impl ContractApi {
    pub async fn get_abi(
        &self,
        api: &mut BscChainApi,
        smart_contarct_address: &str,
    ) -> Result<String, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "contract");
        api.query.add_params("action", "getabi");
        api.query.add_params("address", smart_contarct_address);

        Ok(Response::<String>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?)
    }

    pub async fn get_source_code(
        &self,
        api: &mut BscChainApi,
        smart_contarct_address: &str,
    ) -> Result<Response<Vec<Code>>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "contract");
        api.query.add_params("action", "getsourcecode");
        api.query.add_params("address", smart_contarct_address);

        Ok(Response::<Vec<Code>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await)
    }
}

#[cfg(test)]
mod test {
    use super::*;

    fn create_success() -> BscChainApi {
        BscChainApi::new("YOUR_API_KEY_HERE")
    }

    #[actix_rt::test]
    async fn abi_get() {
        let mut m = create_success();
        let _ = ContractApi
            .get_abi(&mut m, "0xBCfCcbde45cE874adCB698cC183deBcF17952812")
            .await
            .unwrap();
        assert!(true)
    }

    #[actix_rt::test]
    async fn get_source_code() {
        let mut m = create_success();
        assert_eq!(
            ContractApi
                .get_source_code(&mut m, "0xBCfCcbde45cE874adCB698cC183deBcF17952812")
                .await
                .unwrap()
                .result()
                .unwrap()[0]
                .contract_name(),
            "PancakeFactory".to_string()
        );
    }
}
